xquery version "3.0";

module namespace nlLongName         = "http://art-decor.org/ns/nlLongName";

declare variable $nlLongName:uitzonderingen  := doc('../resources/lcn-uitzonderingen.xml');

(:
    This method takes a LOINC concept, translates all axis values,
    then calls {@link #generateLCN(String, String, String, String, String)} 
    to generate the LCN. 
:)
declare function nlLongName:getLongName($lab_concept as element()) as xs:string {
    let $nl_concept         := if ($lab_concept/concept/concept) then $lab_concept/concept/concept else ()
    let $en_concept         := $lab_concept/concept

    let $en_component       := $en_concept/component
    let $en_property        := $en_concept/property
    let $en_system          := $en_concept/system
    let $en_scale           := $en_concept/scale
    let $en_timing          := $en_concept/timing
    let $en_method          := $en_concept/method
    
    (: LOINC uses '-' for empty axes, so for all, replace those with ''  :)
    let $nl_component       := if ($nl_concept/component = '-') then '' else $nl_concept/component
    (: don't translate MFr or SFr to LCN (following Regenstrief, who skip them as well):)
    let $nl_property        := if ($nl_concept/property = '-') then '' else $nl_concept/property
    (: only translate system if it does not equal 'XXX':)
    let $nl_system          := if ($en_system = 'XXX') then '' 
                              else if ($nl_concept/system = '-') then '' else $nl_concept/system
    (: only translate timing if it does not equal 'Pt'. :)
    let $nl_timing          := if ($en_timing = 'Pt') then '' 
                              else if ($nl_concept/timing = '-') then '' else $nl_concept/timing
    (: Scale 'Nar' -> (tekstueel), 'Doc' -> (document) :)
    let $nl_scale           := if ($en_scale = 'Nar') then ' (tekstueel)' 
                              else if ($en_scale = 'Doc') then ' (document)' else ()
    let $nl_method          := if ($nl_concept/method = '-') then '' else $nl_concept/method

    (: do uitzondering for properties :)
    let $uitzondering       := $nlLongName:uitzonderingen//property[@en=$en_property]/@nl/string()
    let $nl_property        := if ($uitzondering) then $uitzondering else $nl_property
    
    (: do uitzondering for methods :)
    let $uitzondering       := $nlLongName:uitzonderingen//method[@en=$en_method]/@nl/string()
    let $nl_method        := if ($uitzondering) then $uitzondering else $nl_method

    let $lcn            := 
                        if ($nl_component) 
                        then nlLongName:generateLcnFromAxes($nl_component, $nl_property, $nl_system, $nl_scale, $nl_timing, $nl_method) 
                        else 'Geen component beschikbaar'
	return $lcn
};

(:
	This method generates the Dutch LCN from the axis values <b>which have already been translated
	to Dutch!</b>
	
	Format: {component} [{property}] in {system} d.m.v. {method}
	Exceptions:
		- different translations of properties and methods
	 - system XXX, timing Pt and certain properties are left out of the translation
	 
	$component	Dutch component translation
	$property Dutch property translation, or null if it should be left out
	$system Dutch system translation, or null if it should be left out
	$timing Dutch timing translation (at present usually identical to English: if this changes this method should be revised!)
	$method Dutch method translation, or null if no method was specified
	@return the LCN
:)
declare function nlLongName:generateLcnFromAxes($component as xs:string, $property as xs:string?, $system as xs:string?, $scale as xs:string?, $timing as xs:string?, $method as xs:string?) as xs:string {
    let $system       := if ($system = 'XXX' or $system = ()) then '' else $system
    let $property     := if ($property = 'XXX' or $property = ()) then '' else $property
    let $timing       := if ($timing = 'XXX' or $timing = 'Moment' or $timing = 'moment' or $timing = 'Pt' or $timing = ()) then '' else $timing
    let $method       := if ($method = ()) then '' else $method
    (: First add the component :)
    let $lcn          := nlLongName:getProperCasedValue($component)
    let $lcn          := 
        if ($property) 
        (:then, if the property has been translated, add the translation between angular brackets:)
        then concat($lcn, ' [', nlLongName:getProperCasedValue($property), ']')
        else $lcn 
    let $urineTiming  := if ($timing) then nlLongName:getTimingNr($timing) else ''
    let $lcn          :=
        if (lower-case($system) = 'urine' and $timing)
        then concat($lcn, ' in ', nlLongName:getProperCasedValue($urineTiming), '-uursurine')
        (: urine samples with particular timings are written as a compound (special case) :)
        else 
        (: in all other cases add first the system :)
            (: If no system, systempart is empty :)
            let $systempart := 
                if ($system)
                then concat(' in ', nlLongName:getProperCasedValue($system))
                else ''
            let $timingpart := 
                if ($timing)
                then 
                    let $timingNr       := replace($timing, '[^0-9]', '')
                    let $timingAlpha    := replace($timing, '[0-9]', '')
                    let $timingNL :=
                        switch ($timingAlpha)
                        case ('M') return if ($timingNr = '1') then 'minuut' else 'minuten'
                        case ('H') return 'uur'
                        case ('U') return 'uur'
                        case ('D') return if ($timingNr = '1') then 'dag' else 'dagen'
                        case ('W') return if ($timingNr = '1') then 'week' else 'weken'
                        case ('Mo') return if ($timingNr = '1') then 'maand' else 'maanden'
                        case ('Y') return 'jaar'
                        (: Emtpy sequence in case of unexpected values :)
                        default return ''
                    return
                        if ($timingNr = '' or $timingNL = '')
                        (: Return original in case of unexpected values :)
                        then nlLongName:getProperCasedValue($timing)
                        else concat (' gedurende ', $timingNr, ' ', $timingNL)
                (: If no timing, timingpart is empty :)
                else ''
            return concat($lcn, concat($systempart, ' ', $timingpart))
    let $lcn    := 
        if ($method)
        then concat($lcn, concat(' d.m.v. ', nlLongName:getProperCasedValue($method)))
        else $lcn
    let $lcn    := 
        if ($scale)
        then concat($lcn, $scale)
        else $lcn
    (: et the first letter of the LCN to upper case and remove any trailing whitespace :)
    let $lcn    := normalize-space(nlLongName:setFirstCapital($lcn))
    return $lcn
};


(:
	Sets the first letter of the Long Common Name to upper case. Please note that the
	first token is not necessarily the first letter! So we go forward through the string 
	until we find a letter.
:)
declare function nlLongName:setFirstCapital($string) {
    if (string-length($string) < 1)
    then ()
    else
    if (starts-with($string, 'pH')) then $string else
    if (starts-with(lower-case($string), 'ij')) then concat('IJ', substring($string, 3)) else
        let $first := substring($string, 1, 1)
        let $rest := substring($string, 2)
        return
            if (matches($first, '[a-zA-Z]'))
            then concat(upper-case($first), $rest)
            else concat($first, nlLongName:setFirstCapital($rest)) 
};

(:
	Extracts the number in a 'timing' expression. E.g. '24' in '24H'
	This method is (only) used to generate the translation of 24H urine specimen.
:)
declare function nlLongName:getTimingNr($timing as xs:string) {
		if ($timing = ()) then
			()
        else
            string-join(
                for $char at $idx in (1 to string-length($timing))
                let $str := substring($timing, $idx, 1)
                return if (number($str)) then $str else ()
                ,
                ''
            )
};

(:
	Gets the value in lower case, unless it is an acronym
	Always use this method to append a translated axis. That way you remove the capitals
	that you do not need but keep the significant ones (in acronyms).
:)
declare function nlLongName:getProperCasedValue($value as xs:string) as xs:string {
    (: split by whitespace into tokens (words) (this will only yield non-empty token, no need to test for empty ones) :)
    let $tokens := tokenize(normalize-space($value), ' ')
    (: The first one is lower-cased, except for acronyms :)
    let $tokens := 
        (
        for $token in $tokens[1]
        return 
            if (nlLongName:isAcronym($token)) then
                $token
            else
                lower-case($token)
         ,
        $tokens[position() > 1]
         )
    (: String-join all components (string + all tokens) on space :)
    let $result := string-join($tokens, ' ')
    (: Collapse all whitespace to single space, trim from start and end :)
    return normalize-space($result)
};

(:
	Checks if the given string is an acronym, by checking for capital letters. The first character of the first
	token can be upper case, all others are considered as indicating an acronym. We make an exception for tokens with a digit 
	at the start, e.g. 5-Aminoimidazole-4-carboxamide
	 
	$token A token (word)
:)

declare function nlLongName:isAcronym($token as xs:string) as xs:boolean {
        (:  if the token is one character long, keep capitals as they are, 
            e.g. F in hemoglobin F :)
		if (string-length($token) = 1) then							    
			true()

        (: if the token starts with a digit, add in lower case, 
		   (probably substance name, e.g. 5-Aminoimidazole-4-carboxamide :)
        else if (number(substring($token, 1, 1)) = (1 to 9)) then
		  false()
		
	    (: the first letter of the first token is always a capital, also for not-acronyms, so skip that
	       if string converted to lower-case is not equal to string, then it contains upper-case letters 
	       (digits and non-alfa chars are untouched by lower-case() :)
	    else if (lower-case(substring($token, 2)) != substring($token, 2)) then 
            true()
        else
        (: no other capital letters found, so not an acronym :)
            false()													
};

declare function nlLongName:acronyms_ok() as xs:boolean {
    if (
        nlLongName:isAcronym('A') = true()
        and
        nlLongName:isAcronym('5-Aminoimidazole-4-Carboxamide') = false()
        and
        nlLongName:isAcronym('IgE') = true()
        and
        nlLongName:isAcronym('Massa concentratie') = false()
        )
    then true()
    else false()
};

declare function nlLongName:getProperCasedValue_ok() as xs:boolean {
    if (
        nlLongName:getProperCasedValue('in urine') = 'in urine'
        and
        nlLongName:getProperCasedValue('in Urine') = 'in Urine'
        and
        nlLongName:getProperCasedValue('in UrinE') = 'in UrinE'
        and
        nlLongName:getProperCasedValue('    in    heel     veel     urine') = 'in heel veel urine' 
        and
        nlLongName:getProperCasedValue('16-Beta,18-dihydroxydehydro-epi-androsteron') = '16-beta,18-dihydroxydehydro-epi-androsteron'
       )
    then true() 
    else false()
};

declare function nlLongName:setFirstCapital_ok() as xs:boolean {
    if (
        nlLongName:setFirstCapital('123ab') = '123Ab'
        and
        nlLongName:setFirstCapital('Feikje') = 'Feikje'
        and
        nlLongName:setFirstCapital('feikje') = 'Feikje'
        and
        nlLongName:setFirstCapital('fe12j3') = 'Fe12j3'
        and
        nlLongName:setFirstCapital('123') = '123'
        )
    then true()
    else false()
};

(:declare function nlLongName:translateValue_ok() as xs:boolean {
    let $labconcept1 := labterm:getLabConceptById('10961-1', 'nl-NL', true(), '')
    let $labconcept2 := labterm:getLabConceptById('7014-4', 'nl-NL', true(), '')
    return if (
        nlLongName:translateValue($labconcept1, 'SYSTEM') = 'Serum'
        and
        nlLongName:translateValue($labconcept1, 'TIME_ASPCT') = ''
        and
        nlLongName:translateValue($labconcept1, 'PROPERTY') = 'eenheden/volume'
        and
        nlLongName:translateValue($labconcept2, 'METHOD_TYP') = "gradiëntstrip"
        ) 
        then true()
        else false()
};
:)
declare function nlLongName:test_all() as xs:boolean* {
    (
        nlLongName:acronyms_ok(),
        nlLongName:getProperCasedValue_ok(),
        nlLongName:setFirstCapital_ok()
(:        ,
        nlLongName:translateValue_ok():)
    )
};