xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)

module namespace utilsc                 = "http://art-decor.org/ns/api/util-scenario";

import module namespace setlib          = "http://art-decor.org/ns/api/settings" at "settings-lib.xqm";
import module namespace utillib         = "http://art-decor.org/ns/api/util" at "util-lib.xqm";
import module namespace decorlib        = "http://art-decor.org/ns/api/decor" at "decor-lib.xqm";
import module namespace histlib         = "http://art-decor.org/ns/api/history" at "history-lib.xqm";
import module namespace ruleslib        = "http://art-decor.org/ns/api/rules" at "rules-lib.xqm";

import module namespace errors          = "http://e-editiones.org/roaster/errors";

(:~ All functions support their own override, but this is the fallback for the maximum number of results returned on a search :)
declare %private variable $utilsc:maxResults             := 50;
declare %private variable $utilsc:ITEM-STATUS            := utillib:getDecorTypes()/ItemStatusCodeLifeCycle;
declare %private variable $utilsc:STATUSCODES-FINAL      := ('final', 'pending', 'rejected', 'cancelled', 'deprecated');
declare %private variable $utilsc:VALUEDOMAIN-TYPE       := utillib:getDecorTypes()/DataSetValueType;
declare %private variable $utilsc:ACTOR-TYPE             := utillib:getDecorTypes()/ScenarioActorType;
declare %private variable $utilsc:TRANSACTION-TYPE       := utillib:getDecorTypes()/TransactionType;
declare %private variable $utilsc:ACTOR-ROLE             := utillib:getDecorTypes()/ActorType;
declare %private variable $utilsc:STATUSCODES-NONDEPRECS := ('draft', 'final', 'new');

(: =========================================================================================================================================== :)
(: ===============                                              SCENARIO LOGIC                                                 =============== :)
(: =========================================================================================================================================== :)

(:~ Central logic retrieving DECOR scenario based on $id (oid) and optionally $effectiveDate (yyyy-mm-ddThh:mm:ss) denoting its version
    @param $id                      - required parameter denoting the id of the scenario
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the scenario. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utilsc:getScenario($id as xs:string, $effectiveDate as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?) {
        
    let $scenario               := utillib:getScenario($id, $effectiveDate, $projectVersion[1], $projectLanguage[1])
    
    for $sc in $scenario
    return
        element {name($sc)} {
            $sc/@*,
            $sc/name,
            $sc/desc,
            if ($sc[publishingAuthority]) then $sc/publishingAuthority else utillib:inheritPublishingAuthority($scenario),
            if ($sc[copyright]) then $sc/copyright else utillib:inheritCopyright($scenario),
            $sc/(* except (name | desc | publishingAuthority | copyright | transaction))
            ,
            (: we return just a count of issues and leave it up to the calling party to get those if required :)
            <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $sc/@id][@effectiveDate = $sc/@effectiveDate])}"/>
            ,
            for $trg in $sc/transaction
            return
                element {name($trg)} {
                $trg/@*,
                $trg/name,
                $trg/desc,
                if ($trg[publishingAuthority]) then $trg/publishingAuthority else utillib:inheritPublishingAuthority($trg),
                if ($trg[copyright]) then $trg/copyright else utillib:inheritCopyright($trg),
                $trg/(* except (name | desc | publishingAuthority | copyright | transaction)),
                for $trl in $trg/transaction
                return
                    element {name($trl)} {
                    $trl/@*,
                    $trl/name,
                    $trl/desc,
                    if ($trl[publishingAuthority]) then $trl/publishingAuthority else utillib:inheritPublishingAuthority($trl),
                    if ($trl[copyright]) then $trl/copyright else utillib:inheritCopyright($trl),
                    $trl/(* except (name | desc | publishingAuthority | copyright))
                }
            }
        }
};

(:~ Central logic returning object with zero or more child scenarios from project
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $datasetId               - optional. Filter output to just those based on this dataset-id
    @param $datasetEffectiveDate    - optional. null filters to the newest version based on $datasetId, yyyy-mm-ddThh:mm:ss for this specific version
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return list with actors
    @since 2020-05-03
:)
declare function utilsc:getScenarioList($projectPrefixOrId as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $datasetId as xs:string?, $datasetEffectiveDate as xs:string?, $sort as xs:string?, $sortorder as xs:string?, $max as xs:integer?) {

    let $decor                  := utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $allcnt                 := count($decor/scenarios/scenario)
    
    let $filteredScenarios      := if (empty($datasetId)) then $decor/scenarios/scenario else utilsc:getScenariosByDataset($datasetId, $datasetEffectiveDate, $projectPrefixOrId, $projectVersion, $projectLanguage)
    (: only want to offer scenarios from our own project. If someone added a dataset from our project 
        onto his transaction, we would have scenarios from other projects :)
    let $filteredScenarios      := $filteredScenarios[ancestor::decor/project[@prefix = $decor/project/@prefix]]

    let $transactionmap         :=
        map:merge(
            if (empty($datasetId)) then (
                for $key in distinct-values($filteredScenarios/transaction/concat(@id, @effectiveDate))
                return map:entry($key, ())
            )
            else (
                for $key in distinct-values(utillib:getTransactionsByDataset($datasetId, $datasetEffectiveDate)/ancestor-or-self::transaction[parent::scenario]/concat(@id, @effectiveDate))
                return map:entry($key, ())
            )
        )
    let $projectLanguage      := 
        if (empty($projectLanguage)) then 
            if (empty($projectPrefix)) then $filteredScenarios/ancestor::decor/project/@defaultLanguage else $decor/project/@defaultLanguage
        else $projectLanguage
    
    (: build map of template attributes. is more costly when you do that deep down :)
    let $templatemap            := 
        for $sc in $filteredScenarios
        let $pfx                := $sc/ancestor::decor/project/@prefix
        group by $pfx
        return utilsc:buildTemplateMap($sc, $sc[1]/ancestor::decor)
    
    let $filteredScenarios      :=
        for $sc in $filteredScenarios
        return
        element {name($sc)} {
            $sc/@*,
            $sc/name,
            $sc/desc,
            if ($sc[publishingAuthority]) then $sc/publishingAuthority else utillib:inheritPublishingAuthority($sc),
            if ($sc[copyright]) then $sc/copyright else utillib:inheritCopyright($sc),
            $sc/(* except (name | desc | publishingAuthority | copyright | transaction)),
            for $trg in $sc/transaction
            return
                element {name($trg)} {
                $trg/@*,
                $trg/name,
                $trg/desc,
                if ($trg[publishingAuthority]) then $trg/publishingAuthority else utillib:inheritPublishingAuthority($trg),
                if ($trg[copyright]) then $trg/copyright else utillib:inheritCopyright($trg),
                $trg/(* except (name | desc | publishingAuthority | copyright | transaction)),
                for $trl in $trg/transaction
                return
                    element {name($trl)} {
                    $trl/@*,
                    $trl/name,
                    $trl/desc,
                    if ($trl[publishingAuthority]) then $trl/publishingAuthority else utillib:inheritPublishingAuthority($trl),
                    if ($trl[copyright]) then $trl/copyright else utillib:inheritCopyright($trl),
                    $trl/(* except (name | desc | publishingAuthority | copyright))
                }
            }
         }

    (:can sort on indexed db content -- faster:)
    let $results                :=
        switch ($sort) 
        case 'effectiveDate' return 
            if (empty($sortorder)) then 
                for $scenario in $filteredScenarios order by $scenario/@effectiveDate return $scenario
            else
                for $scenario in $filteredScenarios order by $scenario/@effectiveDate descending return $scenario
        case 'name' return 
            if (empty($sortorder)) then 
                for $scenario in $filteredScenarios order by lower-case(($scenario/name[@language = $decor/project/@defaultLanguage], $scenario/name)[1]) return $scenario
            else
                for $scenario in $filteredScenarios order by lower-case(($scenario/name[@language = $decor/project/@defaultLanguage], $scenario/name)[1]) descending return $scenario
        default return $filteredScenarios
    
    let $count                  := count($results)
    let $max                    := if ($max ge 1) then $max else $utilsc:maxResults
    
    return
    <list artifact="SC" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
    {
        for $scenario in $results
        return utilsc:scenarioBasics($scenario, $transactionmap, $projectPrefix, $projectLanguage[1], $templatemap)
    }
    </list>
};

(:~ Central logic for creating scenario, new scenario with its own id and effectiveDate based on defaultBaseId for scenarios

    Situation 1: completely new scenario with its own id and effectiveDate
    - Input MAY be "empty" then we create a single scenario with id/effectiveDate/empty name and desc/single transaction group containing single transaction
    - Input MAY be a scenario without id/effectiveDate but with name/desc and other direct properties. If there are no transactions, we do the same as for empty input. 
    - Transaction handling is sent to transaction api where a similar function works in a similar way
    
    Situation 2: new scenario version based on existing scenario with new id and effectiveDate
    - Parameter sourceId and sourceEffectiveDate are required. Input SHALL be empty
    - Return a copy of existing scenario with new id and effectiveDates on scenario and child transactions 
    
    Note: if you want to update a transaction (group) from one dataset to the other, you can do that by adding 
          transaction.representingTemplate.targetDataset and optionally .targetDatasetFlexibility next to the original 
          transaction.representingTemplate.sourceDataset and .sourceDatasetFlexibility 
    
    The scenario creation process then tries to find matches in the new dataset for the concepts listed from the old dataset. This logic is handed off to the transaction api  
    
    @param $bearer-token        - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $projectPrefix       - required. what project to create into             
    @param $sourceId            - optional. the scenario id to create the new scenario from 
    @param $sourceEffectiveDate - optional. the scenario effectiveDate to create the scenario from 
    @param $request-body        - optional. body containing new scenario structure
    @return scenario structure
    @since 2020-05-03
:)
declare function utilsc:postScenario($authmap as map(*), $projectPrefixOrId as xs:string, $sourceScenarioId as xs:string?, $sourceScenarioEffectiveDate as xs:string?, $data as element(scenario)?) as element(scenario) {
    
    let $decor                  := utillib:getDecor($projectPrefixOrId)
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project ''', $projectPrefixOrId, ''', does not exist. Cannot create in non-existent project.'))
        )
    
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $defaultLanguage        := $decor/project/@defaultLanguage
    let $storedScenario         := if (empty($sourceScenarioId)) then () else utillib:getScenario($sourceScenarioId, $sourceScenarioEffectiveDate)
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  :=
        if (empty($sourceScenarioId)) then () else if ($storedScenario) then () else (
            error($errors:BAD_REQUEST, concat('Source scenario id ''', $sourceScenarioId, ''' effectiveDate ''', $sourceScenarioEffectiveDate, ''' requested but not found. Could not create new scenario from source scenario.'))
        )
    
    let $newId                  := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-SCENARIO, decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-SCENARIO)/@id)/@id
    let $newEffectiveDate       := format-date(current-date(), '[Y0001]-[M01]-[D01]') || 'T00:00:00'
    let $tridnext               := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TRANSACTION, decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TRANSACTION)/@id)
    
    (: guarantee defaultLanguage. Preserve any incoming other languages. Guarantee name. Leave desc optional :)
    let $data                   := if ($storedScenario) then $storedScenario else $data
    let $newScenario            :=
        <scenario id="{$newId}" effectiveDate="{$newEffectiveDate}" statusCode="draft"> 
        {
            $data/@versionLabel[string-length() gt 0]
            (: don't copy expirationDate, officialReleaseDate, canonicalUri :),
            attribute lastModifiedDate {$newEffectiveDate},
            utilsc:handleScenarioElements($data, $defaultLanguage, exists($storedScenario))
            ,
            if ($data/transaction) then (
                (: TODO: handoff to transaction function :)
                for $tr at $i in $data/transaction
                let $trid     := $tridnext/@base || '.' || xs:integer($tridnext/@max) + $i + count($tr/preceding-sibling::transaction/transaction)
                return
                <transaction id="{$trid}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="group">
                {
                    attribute lastModifiedDate {$newEffectiveDate},
                    utilsc:handleTransactionElements($tr, $defaultLanguage)
                    ,
                    for $trc at $j in $tr/transaction
                    let $trid   := $tridnext/@base || '.' || xs:integer($tridnext/@max) + $i + count($tr/preceding-sibling::transaction/transaction) + $j
                    return
                        <transaction id="{$trid}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="{($trc/@type, 'stationary')[1]}">
                        {
                            $trc/@model,
                            attribute lastModifiedDate {$newEffectiveDate},
                            utilsc:handleTransactionElements($trc, $defaultLanguage)
                        }
                        </transaction>
                }
                </transaction>
            )
            else (
                let $trgroupid  := $tridnext/@id
                let $tr1id      := $tridnext/@base || '.' || xs:integer($tridnext/@next) + 1
                return
                <transaction id="{$trgroupid}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="group" lastModifiedDate="{$newEffectiveDate}">
                    <name language="{$defaultLanguage}"/>
                    <transaction id="{$tr1id}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="stationary" lastModifiedDate="{$newEffectiveDate}">
                        <name language="{$defaultLanguage}"/>
                        <actors/>
                    </transaction>
                </transaction>
            )
        }
        </scenario>
        
    let $updateScenarios        :=
        if ($decor/scenarios[questionnaire | questionnaireresponse]) then
            update insert $newScenario preceding $decor/scenarios/(questionnaire | questionnaireresponse)[1]
        else if ($decor/scenarios) then update insert $newScenario into $decor/scenarios 
        else update insert <scenarios><actors/>{$newScenario}</scenarios> preceding $decor/ids

    return utilsc:getScenario($newScenario/@id, $newScenario/@effectiveDate, (), ())
};

(:~ Central logic for updating an existing scenario
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR scenario/@id to update
    @param $effectiveDate   - required. DECOR scenario/@effectiveDate to update
    @param $data            - required. DECOR scenario xml element containing everything that should be in the updated scenario
    @return concept object as xml with json:array set on elements
:)
declare function utilsc:putScenario($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(scenario), $deletelock as xs:boolean) {

    let $editedScenario         := $data
    let $storedScenario         := utillib:getScenario($id, $effectiveDate)
    let $decor                  := $storedScenario/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if (empty($storedScenario)) then 
            error($errors:BAD_REQUEST, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        else
        if (count($storedScenario) gt 1) then
            error($errors:SERVER_ERROR, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' occurs ' || count($storedScenario) || ' times. Inform your database administrator.')
        else
        if ($storedScenario[@statusCode = $utilsc:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it one of status: ' || string-join($utilsc:STATUSCODES-FINAL, ', '))
        else ()
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  :=
        if ($data[@id = $id] | $data[empty(@id)]) then () else (
            error($errors:BAD_REQUEST, concat('Scenario SHALL have the same id as the scenario id ''', $id, ''' used for updating. Found in request body: ', ($data/@id, 'null')[1]))
        )
    
    let $check                  :=
        if ($data[@effectiveDate = $effectiveDate] | $data[empty(@effectiveDate)]) then () else (
            error($errors:BAD_REQUEST, concat('Scenario SHALL have the same effectiveDate as the scenario effectiveDate ''', $effectiveDate, ''' used for updating. Found in request body: ', ($data/@effectiveDate, 'null')[1]))
        )
    let $check                  :=
        if ($data[@statusCode = $storedScenario/@statusCode] | $data[empty(@statusCode)]) then () else (
            error($errors:BAD_REQUEST, concat('Scenario SHALL have the same statusCode as the scenario statusCode ''', $storedScenario/@statusCode, ''' used for updating. Found in request body: ', ($data/@statusCode, 'null')[1]))
        )
    
    let $check                  := utilsc:checkScenarioTransactionsForUpdate($editedScenario//transactions, $storedScenario)
    
    let $check                  :=
        for $object in $editedScenario[edit] | $editedScenario/descendant::transaction[edit]
        let $storedObject       := $storedScenario[self::scenario][@id = $object/@id][@effectiveDate = $object/@effectiveDate] | $storedScenario/descendant::transaction[@id = $object/@id][@effectiveDate = $object/@effectiveDate]
        return utilsc:checkScenarioObjectsForUpdate($object, $storedObject, $editedScenario, $decor)
    
    (: set and check a lock on every object to be changed :)
    let $check                  :=
        for $object in $editedScenario[edit] | $editedScenario/descendant::transaction[edit]
        let $storedObject       := $storedScenario[self::scenario][@id = $object/@id][@effectiveDate = $object/@effectiveDate] | $storedScenario/descendant::transaction[@id = $object/@id][@effectiveDate = $object/@effectiveDate]
        return if ($object/@id) then utilsc:checkScenarioObjectLock($authmap, $storedObject, false()) else ()
    
    (: save history:)
    let $intention              := if ($storedScenario[@statusCode='final'][@effectiveDate = $editedScenario/@effectiveDate]) then 'patch' else 'version'
    let $history                := if ($storedScenario and $editedScenario/descendant-or-self::*[edit | move]) then histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-SCENARIO, $projectPrefix, $intention, $storedScenario) else ()

    let $baseIdTransaction      := decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TRANSACTION)[1]/@id
    let $nextTransactionId      := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TRANSACTION, $baseIdTransaction)
    let $insert                 := if ($decor/scenarios[@nextTransactionLeaf]) 
        then update value $decor/scenarios/@nextTransactionLeaf with $nextTransactionId/@next
        else update insert attribute nextTransactionLeaf {$nextTransactionId/@next} into $decor/scenarios
        
    let $preparedScenario       := utilsc:prepareScenarioForUpdate($editedScenario, $storedScenario, $baseIdTransaction)
    
    let $check                  := ruleslib:checkDecorSchema($preparedScenario, 'update')

    let $locks                  := if ($deletelock) then decorlib:getLocks($authmap, $storedScenario, true()) else ()
    let $updates                := update replace $storedScenario with $preparedScenario
    let $delete                 := update delete $decor/scenarios/@nextTransactionLeaf
    
    let $delete                 := if ($locks) then update delete $locks else ()
    
    return utilsc:getScenario($preparedScenario/@id, $preparedScenario/@effectiveDate, (), ())
};

(:~ Central logic for patching an existing scenario
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR scenario/@id to update
    @param $effectiveDate   - required. DECOR scenario/@effectiveDate to update
    @param $data            - required. DECOR scenario xml element containing everything that should be in the updated scenario
    @return concept object as xml with json:array set on elements
:)
declare function utilsc:patchScenario($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(scenario) {

    let $storedScenario         := utillib:getScenario($id, $effectiveDate)
    let $decor                  := $storedScenario/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if ($storedScenario) then () else (
            error($errors:BAD_REQUEST, 'Scenario with id ' || $id || ' and effectiveDate ' || $effectiveDate || ' does not exist')
        )
    
    let $lock                   := utilsc:checkScenarioAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedScenario[@statusCode = $utilsc:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilsc:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, concat('Scenario cannot be patched while it has one of status: ', string-join($utilsc:STATUSCODES-FINAL, ', '), if ($storedScenario/@statusCode = 'pending') then 'You should switch back to status draft to edit.' else ()))
            )
        else ()
    
    let $check                  := utilsc:checkScenarioParameters($data, $storedScenario)
    
    let $intention              := if ($storedScenario[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-SCENARIO, $projectPrefix, $intention, $storedScenario)
    
    let $update                 := utilsc:patchScenarioParameters($data, $storedScenario)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedScenario       := utilsc:handleScenario($storedScenario)
    
    let $update                 := update replace $storedScenario with $preparedScenario
    let $update                 := update delete $lock
    
    return utilsc:getScenario($id, $effectiveDate, (), ())
};

(:~ Central logic for patching an existing scenario statusCode, versionLabel, expirationDate and/or officialReleaseDate
   @param $authmap required. Map derived from token
   @param $id required. DECOR concept/@id to update
   @param $effectiveDate required. DECOR concept/@effectiveDate to update
   @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
   @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
   @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
   @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
   @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
   @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
   @return list object with success and/or error elements or error
:)
declare function utilsc:setScenarioStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element()* {

    let $object                 := utillib:getScenario($id, $effectiveDate)
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () else
        if ($object) then
            error($errors:SERVER_ERROR, 'Scenario id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Scenario id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $testUpdate             := utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate 
        else if ($listOnly) then $testUpdate 
        else utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        
    return
        if ($results[self::error] and not($listOnly)) then
            error($errors:BAD_REQUEST, string-join(
                for $e in $results[self::error]
                return
                    $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
                , ' ')
            )
        else $results
};

(:~ Central logic returning list object with zero or more child actors from project
    @param $projectPrefix           - required. limits scope to this project only
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $sort                    - optional parameter to indicate sorting. Only option 'name'
    @param $sortorder               - optional parameter to indicate sort order. Only option 'descending'
    @return list with actors
    @since 2020-05-03
:)
declare function utilsc:getScenarioActorList($projectPrefixOrId as xs:string, $projectVersion as xs:string?, $projectLanguage as xs:string?, $sort as xs:string?, $sortorder as xs:string?, $max as xs:integer?) {

    let $decor                  := utillib:getDecor($projectPrefixOrId, $projectVersion, $projectLanguage)
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $actors                 := $decor/scenarios/actors/actor
    
    let $allcnt                 := count($actors)
    
    (:can sort on indexed db content -- faster:)
    let $results                :=
        switch ($sort) 
        case 'name' return 
            if (empty($sortorder)) then 
                for $actor in $actors order by lower-case(($actor/name[@language = $decor/project/@defaultLanguage], $actor/name)[1]) return $actor
            else
                for $actor in $actors order by lower-case(($actor/name[@language = $decor/project/@defaultLanguage], $actor/name)[1]) descending return $actor
        default return $actors
    
    let $count              := count($results)
    let $max                := if ($max ge 1) then $max else $utilsc:maxResults
    
    return
        <list artifact="AC" current="{if ($count le $max) then $count else $max}" total="{$count}" all="{$allcnt}" lastModifiedDate="{current-dateTime()}" xmlns:json="http://www.json.org">
        {
            subsequence($results, 1, $max)
        }
        </list>

};

(:~ Central logic for retrieving DECOR scenario actor
    @param $projectPrefix            - optional. limits scope to this project only
    @param $id                       - required. id for the actor to retrieve
    @return one or zero actor object
    @since 2020-08-11
:)
declare function utilsc:getScenarioActor($projectPrefix as xs:string?, $id as xs:string) as element(actor)* {
    
    let $actor                  := $setlib:colDecorData//scenarios/actors/actor[@id = $id]
    return if (empty($projectPrefix)) then $actor else $actor[ancestor::decor/project[@prefix = $projectPrefix]]
};

(:~ Central logic for creating scenario actor
    @param $authmap      - required. Map derived from token
    @param $prefix       - required. DECOR project/@prefix
    @param $data         - required. DECOR actor object as xml
    @return actor object as xml
:)
declare function utilsc:postScenarioActor($authmap as map(*), $projectPrefixOrId as xs:string, $data as element(actor)?) as element(actor) {
    
    let $decor                  := utillib:getDecor($projectPrefixOrId)
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project ''', $projectPrefixOrId, ''', does not exist. Cannot create in non-existent project.'))
        )
    
    let $projectPrefix          := ($decor/project/@prefix)[1]
    let $defaultLanguage        := $decor/project/@defaultLanguage
    let $actorType              := ($data/@type, $utilsc:ACTOR-TYPE/enumeration/@value)[1]
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  :=
        if ($actorType = $utilsc:ACTOR-TYPE/enumeration/@value) then () else (
            error($errors:BAD_REQUEST, concat('Actor type ''', $actorType, ''' not supported. Supported are: ', string-join($utilsc:ACTOR-TYPE/enumeration/@value, ', ')))
        )
    
    let $check                  :=
        if ($data[name[text()[not(. = '')]]/@language]) then
            if ($data[count(name/@language) = count(distinct-values(name/@language))]) then () else (
                error($errors:BAD_REQUEST, 'Actor name SHALL be unique in language. Found languages: ' || string-join($data/name/@language, ', '))
            )
        else (
            error($errors:BAD_REQUEST, 'Actor SHALL have at least one non-empty name marked with a language format ll-CC')
        )
    
    let $check                  :=
        if ($data[desc/@language]) then
            if ($data[count(desc/@language) = count(distinct-values(desc/@language))]) then () else (
                error($errors:BAD_REQUEST, 'Actor desc SHALL be unique in language. Found languages: ' || string-join($data/desc/@language, ', '))
            )
        else ()
    
    let $newId                  := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-ACTOR, decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-ACTOR)/@id)/@id
    
    (: guarantee defaultLanguage. Preserve any incoming other languages. Guarantee name. Leave desc optional :)
    let $newActor               :=
        <actor id="{$newId}" type="{$actorType}">
        {
            if ($data/name[@language = $defaultLanguage]) then $data/name[@language = $defaultLanguage][1]
            else (
                <name language="{$defaultLanguage}">
                {
                    data($data/name[1])
                }
                </name>
            ),
            for $lang in distinct-values($data/name/@language[not(. = $defaultLanguage)])
            return $data/name[@language = $lang][1]
            ,
            utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/desc)
        }
        </actor>
        
    let $actorsAvailable    :=
        if ($decor/scenarios/actors) then update insert $newActor into $decor/scenarios/actors
        else if ($decor/scenarios) then update insert <actors>{$newActor}</actors> preceding $decor/scenarios/node()[1]
        else update insert <scenarios><actors>{$newActor}</actors></scenarios> preceding $decor/ids

    return utilsc:getScenarioActor($projectPrefix, $newActor/@id)
};

(:~ Central logic for patching an existing scenario actor type|name|desc

<parameters>
    <parameter op="replace" path="/displayName" value="new value"/>
    <parameter op="replace" path="/priority" value="new value"/>
    <parameter op="add" path="/">
        <value>
            <tracking statusCode="closed">
                <desc language="de-DE">yadadadad</desc>
            </tracking>
        </value>
    </parameter>
</parameters>

    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR actor/@id to update
    @param $projectPrefix   - optional. DECOR project/@prefix to update
    @param $request-body    - required. DECOR xml element parameter elements each containing RFC 6902 compliant contents
    @return actor object as xml with json:array set on elements
:)
declare function utilsc:patchScenarioActor($authmap as map(*), $projectPrefix as xs:string?, $id as xs:string, $data as element()) as element(actor) {

    let $actor                  := utilsc:getScenarioActor($projectPrefix, $id)
    let $decor                  := $actor/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($actor) gt 1) then (
            error($errors:SERVER_ERROR, concat("Found multiple actors for id '", $id, "'. Expected 0..1. You should delete and recreate this actor. Consider alerting your administrator as this should not be possible."))
        ) else
        if ($actor) then () else (
            error($errors:BAD_REQUEST, concat('Actor with id ''', $id, ''' does not exist'))
        )
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  := utilsc:checkScenarioActorParameters($data, $actor)
    
    let $update                 := utilsc:patchScenarioActorParameters($data, $actor)

    return utilsc:getScenarioActor($projectPrefix, $id)
        
};

(: =========================================================================================================================================== :)
(: ===============                                              TRANSACTION LOGIC                                              =============== :)
(: =========================================================================================================================================== :)

(:~ Central logic for retrieving a transaction
    @param $id                      - required parameter denoting the id of the transaction
    @param $effectiveDate           - optional parameter denoting the effectiveDate of the transaction. If not given assumes latest version for id
    @param $projectVersion          - optional parameter to select from a release. Expected format yyyy-mm-ddThh:mm:ss
    @param $projectLanguage         - optional parameter to select from a specific compiled language
    @param $treeonly                - optional boolean parameter to get the tree structure only if true
    @param $fullTree                - optional boolean parameter relevant if $treeonly = 'true' to include absent concepts
    @return as-is or as compiled as JSON
    @since 2020-05-03
:)
declare function utilsc:getTransaction($id as xs:string, $effectiveDate as xs:string?, $projectPrefixOrId as xs:string?, $projectVersion as xs:string?, $projectLanguage as xs:string?, $treeonly as xs:boolean?, $fullTree as xs:boolean?) {
        
    let $result                     := utillib:getTransaction($id, $effectiveDate, $projectPrefixOrId[1], $projectVersion[1], $projectLanguage[1])
    
    (: build map of template attributes. is more costly when you do that deep down :)
    let $templatemap                := if ($result) then utilsc:buildTemplateMap($result, $result/ancestor::decor) else ()
    
    return
    if ($result) then 
        let $representingTemplate   := utilsc:representingTemplateBasics($result/representingTemplate, $templatemap, false())
        let $publishingAuthority    := if ($result[@id]) then if ($result[publishingAuthority]) then $result/publishingAuthority else utillib:inheritPublishingAuthority($result) else ()
        let $copyright              := if ($result[@id]) then if ($result[copyright]) then $result/copyright else utillib:inheritCopyright($result) else ()

        return
        if ($treeonly) then 
            for $tr in utillib:getDatasetTree((), (), $id, $effectiveDate, $fullTree)
            return
                element {name($tr)} {
                    $tr/@*,
                    (: copy all template attributes that we do not already have in the $tr/@* set :)
                    for $att in $representingTemplate/@*[starts-with(name(), 'template')]
                    return if ($tr/@*[name() = name($att)]) then () else $att
                    ,
                    $tr/name,
                    $tr/desc,
                    $publishingAuthority,
                    $copyright,
                    $tr/(* except (name | desc | publishingAuthority | copyright | transaction | representingTemplate))
                    ,
                    (: we return just a count of issues and leave it up to the calling party to get those if required :)
                    <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $tr/@id][@effectiveDate = $tr/@effectiveDate])}"/>
                }
        else (
            for $tr in $result
            return
                element {name($tr)} {
                    $tr/@*,
                    $tr/name,
                    $tr/desc,
                    $tr/(* except (name | desc | publishingAuthority | copyright | transaction | representingTemplate))
                    ,
                    (: we return just a count of issues and leave it up to the calling party to get those if required :)
                    <issueAssociation count="{count($setlib:colDecorData//issue/object[@id = $tr/@id][@effectiveDate = $tr/@effectiveDate])}"/>
                    ,
                    $tr/transaction
                    ,
                    (: just create a representingTemplate element if this is a leaf transaction, i.e. has a parent transaction :)
                    if ($tr/parent::transaction) then utilsc:representingTemplateBasics($tr/representingTemplate, $templatemap, true()) else ()
                }
            )
        else ()
    
};

(:~ Central logic for creating transaction, new transaction with its own id and effectiveDate based on defaultBaseId for transactions

Situation 1: completely new transaction with its own id and effectiveDate
- Input MAY be "empty" then we create a single transaction of the requested type and if this is a group containing single transaction
- Input MAY be a transaction without id/effectiveDate but with name/desc and other direct properties. If there are no child transactions, we do the same as for empty input. 

Situation 2: new transaction version based on existing transaction with new id and effectiveDate
- Parameter sourceId and sourceEffectiveDate are required. Input SHALL be empty
- Return a copy of existing transaction with new id and effectiveDates on transaction and child transactions 

Note: if you want to update a transaction (group) from one dataset to the other, you can do that by adding 
      transaction.representingTemplate.targetDataset and optionally .targetDatasetFlexibility next to the original 
      transaction.representingTemplate.sourceDataset and .sourceDatasetFlexibility 

The transaction creation process then tries to find matches in the new dataset for the concepts listed from the old dataset. This logic is handed off to the transaction api  

    @param $scenarioId              - required. scenario id to insert transaction into.
    @param $scenarioEffectiveDate   - required. scenario effectiveDate to insert transaction into.
    @param $transactionBaseId       - optional. baseId to create the new transaction id out of. Defaults to defaultBaseId for type TR if omitted
    @param $transactionType         - optional if request-body is provided. 'group' or 'stationary' or 'initial' or 'back'
    @param $insertMode              - required. 'into' or 'preceding' or 'following'. For preceding and for following, $insertRef is required
    @param $insertRef               - optional. transaction/@id reference for insert. Inserts as new transaction in scenario if empty
    @param $insertFlexibility       - optional. transaction/@effectiveDate reference for insert. Only relevant when two versions of the same transaction are in the same scenario which is logically highly unlikely
    @param $sourceId                - optional. the transaction id to create the new transaction from 
    @param $sourceEffectiveDate     - optional. the transaction effectiveDate to create the transaction from 
    @param $request-body optional. body containing new transaction structure
    @return created transaction structure
    @since 2020-05-03
:)
declare function utilsc:postTransaction($authmap as map(*), $scenarioId as xs:string, $scenarioEffectiveDate as xs:string, $transactionBaseId as xs:string?, $transactionType as xs:string?, $insertMode as xs:string, $insertRef as xs:string?, $insertFlexibility as xs:string?, $sourceId as xs:string?, $sourceEffectiveDate as xs:string?, $data as element(transaction)?) as element(transaction) {
    
    let $storedScenario         := utillib:getScenario($scenarioId, $scenarioEffectiveDate)
    let $sourceTransaction      := if (empty($sourceId)) then () else utillib:getTransaction($sourceId, $sourceEffectiveDate)
    let $insertTransaction      := 
        if (empty($insertRef)) then () 
        else if (empty($insertFlexibility)) then $storedScenario//transaction[@id = $insertRef] 
        else $storedScenario//transaction[@id = $insertRef][@effectiveDate = $insertFlexibility]
    
    let $decor                  := $storedScenario/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $defaultLanguage        := $decor/project/@defaultLanguage
    let $transactionType        := ($transactionType[not(. = '')], $data/@type)[1]
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Scenario with ''', $scenarioId, ''' and effectiveDate ''', $scenarioEffectiveDate, ''', does not exist. Cannot create in non-existent scenario.'))
        )
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  :=
        if (empty($sourceId)) then () else if ($sourceTransaction) then () else (
            error($errors:BAD_REQUEST, concat('Source transaction id ''', $sourceId, ''' effectiveDate ''', $sourceEffectiveDate, ''' requested but not found. Could not create new transaction from source transaction.'))
        )
    
    let $check                  :=
        if (empty($insertRef)) then () else if ($insertTransaction) then () else (
            error($errors:BAD_REQUEST, concat('Insert transaction id ''', $insertRef, ''' effectiveDate ''', $insertFlexibility, ''' requested but not found. Could not create new transaction ', $insertMode, ' the referenced transaction.'))
        )
    
    let $check                  :=
        switch ($insertMode)
        case 'into' 
        case 'following'
        case 'preceding' return ()
        default return (
            error($errors:BAD_REQUEST, 'Parameter insertMode ' || $insertMode || ' SHALL be one of ''into'', ''following'', or ''preceding''')
        )
    
    let $check                  :=
        if (empty($insertRef)) then if ($insertMode = 'into') then () else (
            error($errors:BAD_REQUEST, 'Parameter insertRef SHALL have a value when insertMode is ' || $insertMode || '. Otherwise we don''t know where to insert')
        )
        else ()
    
    let $check                  :=
        if (empty($transactionBaseId)) then () else (
            let $transactionBaseIds := decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-TRANSACTION)
            return if ($transactionBaseIds[@id = $transactionBaseId]) then () else (
                error($errors:BAD_REQUEST, 'Parameter transactionBaseId SHALL have a value that is declared as baseId with type TR (transactions) in the project. Allowable are ' || string-join($transactionBaseIds/@id, ', '))
            )
        )
    
    let $check                  := if ($data) then ruleslib:checkTransaction($data, $decor, 'create') else ()
    
    let $check                  :=
        if ($insertTransaction) then 
            switch ($insertMode)
            case 'into' return 
                if ($insertTransaction[not(@type = 'group')]) then
                    error($errors:BAD_REQUEST, 'Cannot create a new transaction under a transaction of type ' || $insertTransaction/@type || '. Only type group allows child transactions.')
                else if ($transactionType = 'group') then
                    error($errors:BAD_REQUEST, 'Cannot create a new group under a transaction of type ' || $insertTransaction/@type || '. Transaction groups SHALL NOT be nested.')
                else ()
            default return
                if ($insertTransaction/@type = 'group' and $transactionType = 'group') then ()  
                else if (not($insertTransaction/@type = 'group' or $transactionType = 'group')) then () 
                else (
                    error($errors:BAD_REQUEST, 'Cannot create a new transaction of type ' || $transactionType || ' ' || $insertMode || ' a transaction of type ' || $insertTransaction/@type || '. This combination of types is logically incompatible at the same level.')
                )
        else ()
    
    let $newId                  := decorlib:getNextAvailableIdP($decor, $decorlib:OBJECTTYPE-TRANSACTION, decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-TRANSACTION)/@id)
    let $newEffectiveDate       := format-date(current-date(), '[Y0001]-[M01]-[D01]') || 'T00:00:00'
    
    (: guarantee defaultLanguage. Preserve any incoming other languages. Guarantee name. Leave desc optional :)
    let $data                   := if ($sourceTransaction) then $sourceTransaction else $data
    
    let $newTransaction         :=
        if ($data) then (
            <transaction id="{$newId/@id}" effectiveDate="{$newEffectiveDate}" statusCode="draft">
            {
                attribute type {if ($data/transaction) then 'group' else $data/@type},
                $data/@model[string-length() gt 0],
                if ($data/@label[string-length() gt 0]) then attribute label {$data/@label || '-' || substring($newEffectiveDate, 1, 10)} else ()
                (: don't copy expirationDate, officialReleaseDate, canonicalUri :)
                ,
                attribute lastModifiedDate {$newEffectiveDate},
                utilsc:handleTransactionElements($data, $defaultLanguage)
                ,
                for $trc at $j in $data/transaction
                let $trid       := $newId/@base || '.' || xs:integer($newId/@max) + 1 + $j
                return
                    <transaction id="{$trid}" effectiveDate="{$newEffectiveDate}" statusCode="draft">
                    {
                        attribute type {if ($trc/transaction) then 'group' else ($trc/@type[not(. = 'group')], 'stationary')[1]},
                        $trc/@model[string-length() gt 0],
                        if ($trc/@label[string-length() gt 0]) then attribute label {$trc/@label || '-' || substring($newEffectiveDate, 1, 10)} else (),
                        attribute lastModifiedDate {$newEffectiveDate},
                        utilsc:handleTransactionElements($trc, $defaultLanguage)
                    }
                    </transaction>
            }
            </transaction>
        )
        else (
            let $trgroupid  := $newId/@id
            let $tr1id      := $newId/@base || '.' || xs:integer($newId/@next) + 1
            return
            <transaction id="{$trgroupid}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="group" lastModifiedDate="{$newEffectiveDate}">
                <name language="{$defaultLanguage}"/>
                <transaction id="{$tr1id}" effectiveDate="{$newEffectiveDate}" statusCode="draft" type="stationary" lastModifiedDate="{$newEffectiveDate}">
                    <name language="{$defaultLanguage}"/>
                    <actors/>
                </transaction>
            </transaction>
        )
        
    let $insert     :=
        if ($insertTransaction) then
            switch ($insertMode)
            case 'following' return update insert $newTransaction following $insertTransaction[last()]
            case 'preceding' return update insert $newTransaction preceding $insertTransaction[1]
            default          return update insert $newTransaction into $insertTransaction[1]
        else if ($storedScenario[transaction]) then update insert $newTransaction following $storedScenario/transaction[last()]
        else update insert $newTransaction into $storedScenario

    return $newTransaction
};

(:~ Central logic for patching an existing transaction
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR transaction/@id to update
    @param $effectiveDate   - required. DECOR transaction/@effectiveDate to update
    @param $data            - required. DECOR transaction xml element containing everything that should be in the updated transaction
    @return transaction object 
:)
declare function utilsc:patchTransaction($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(parameters)) as element(transaction) {

    let $storedTransaction      := utillib:getTransaction($id, $effectiveDate)
    let $decor                  := $storedTransaction/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix

    let $check                  :=
        if (empty($storedTransaction)) then 
            error($errors:BAD_REQUEST, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        else
        if (count($storedTransaction) = 1) then () else (
            error($errors:SERVER_ERROR, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' occurs ' || count($storedTransaction) || ' times. Inform your database administrator.')
        )
    
    let $lock                   := utilsc:checkScenarioAccess($authmap, $decor, $id, $effectiveDate, true())
    
    let $check                  :=
        if ($storedTransaction[@statusCode = $utilsc:STATUSCODES-FINAL]) then
            if ($data/parameter[@path = '/statusCode'][@op = 'replace'][not(@value = $utilsc:STATUSCODES-FINAL)]) then () else 
            if ($data[count(parameter) = 1]/parameter[@path = '/statusCode']) then () else (
                error($errors:BAD_REQUEST, 'Transaction  with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be patched while it has one of status: ' || string-join($utilsc:STATUSCODES-FINAL, ', '))
            )
        else ()
    
    let $check                  := utilsc:checkTransactionParameters($data, $storedTransaction, $decor)
    
    let $intention              := if ($storedTransaction[@statusCode = 'final']) then 'patch' else 'version'
    let $update                 := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TRANSACTION, $projectPrefix, $intention, $storedTransaction)
    
    let $update                 := utilsc:patchTransactionParameters($data, $storedTransaction)
    
    (: after all the updates, the XML Schema order is likely off. Restore order :)
    let $preparedTransaction    := utilsc:handleTransaction($storedTransaction)
    
    let $update                 := update replace $storedTransaction with $preparedTransaction
    let $update                 := update delete $lock
    
    return $preparedTransaction
};

(:~ Central logic creating a scheduled task for a questionnaire-transform-request 
    @param $bearer-token             - required. provides authorization for the user. The token should be on the X-Auth-Token HTTP Header
    @param $project                  - required. what project to create into
    @param $baseId                  - required. specific baseId to be used for the Questionnaires 
    @param $request-body             - required. body containing new scenario structure
    @return questionnaire-transform-request
    @since 2020-05-08
:)
declare function utilsc:postTransactionQuestionnaire($authmap as map(*), $project as item(), $baseId as xs:string?, $data as element())  as element() {

    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    let $projectLanguage        := $decor/project/@defaultLanguage
    let $now                    := substring(xs:string(current-dateTime()), 1, 19)
    
    let $check                  :=
        if (empty($decor)) then
            error($errors:BAD_REQUEST, 'Cannot find DECOR project ' || $project || ' to write the Questionnaire(s) into.')
        else ()
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $check                  :=
        if (empty($data/transaction)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have one or more transactions')
        else ()
    
    let $check                  :=
        if (count($data/transaction[@id][@effectiveDate]) != count($data/transaction)) then
            error($errors:BAD_REQUEST, 'All transactions SHALL have an id or effectiveDate')       
        else ()
    
    (: The user may request a specific baseId to be used for the Questionnaires. This baseId SHALL be declared as a baseId in the project with type QQ in that case. 
        If the user did not request a specific baseId, then the defaultBaseId for QQ should be assumed. note that QQ as type may not be present in a given project :)
    let $baseId                 := if (empty($baseId)) 
        then decorlib:getDefaultBaseIdsP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE)/@id 
        else $baseId[. = decorlib:getBaseIdsP($decor, $decorlib:OBJECTTYPE-QUESTIONNAIRE)/@id]
        
    let $check                  :=
        if (empty($baseId)) then 
            error($errors:SERVER_ERROR, 'Project ''' || $project || ''' is missing a default base id for type Questionnaire (' || $decorlib:OBJECTTYPE-QUESTIONNAIRE || '). Please work with a decor-admin for your project to add it and QuestionnaireResponse (' || $decorlib:OBJECTTYPE-QUESTIONNAIRERESPONSE ||') first.')
        else ()
    
    return
        for $transaction in $data/transaction
        let $uuid               := util:uuid($transaction)
        let $transformation-request := 
            <questionnaire-transform-request uuid="{$uuid}" for="{$projectPrefix}" on="{current-dateTime()}" by="{$authmap?name}" baseId="{$baseId}" progress="Added to process queue ..." progress-percentage="0">
                {$transaction}
            </questionnaire-transform-request>
        let $write              := xmldb:store($setlib:strDecorScheduledTasks, $projectPrefix || replace($now, '\D', '') || '-' || $uuid || '.xml', $transformation-request)
        let $tt                 := sm:chmod(xs:anyURI($write), 'rw-rw----')   
        return $transformation-request

};

declare function utilsc:process-questionnairetransform($decor as element(decor)?, $fullDatasetTree as element()?, $transaction as element(transaction), $qqid as xs:string) as element() {
    
    let $newEd                      := substring(string(current-dateTime()), 1, 19)
    let $canonicalUri               := utillib:getServerURLFhirCanonicalBase() || 'Questionnaire/'|| $qqid || '--' || replace($newEd, '\D','')

    return (
    <return>
    {
        comment { ' ' || replace($transaction/name[1], '--', '-') || ' ' },
        <questionnaireAssociation questionnaireId="{$qqid}" questionnaireEffectiveDate="{$newEd}">
        {
            for $c in $fullDatasetTree//concept[not(ancestor::conceptList | ancestor-or-self::concept[@maximumMultiplicity[. = '0'] | @conformance[. = 'NP']])]
            return
                (: NB keep value of @elementId in sync with utilsc:process-transaction-concept($concept) for @linkId :)
                <concept ref="{$c/@id}" effectiveDate="{$c/@effectiveDate}" elementId="{$c/@id}"/>
        }
        </questionnaireAssociation>
        ,
        <questionnaire id="{$qqid}" effectiveDate="{$newEd}" statusCode="draft" lastModifiedDate="{$newEd}" canonicalUri="{$canonicalUri}">
        {
            (: we cannot copy the canonicalUri from the transaction. we 'could' generate it here as appropriate for a Questionnaire :)
            $transaction/name,
            $transaction/desc,
            <relationship type="DRIV" ref="{$transaction/@id}" flexibility="{$transaction/@effectiveDate}"/>,
            $transaction/publishingAuthority,
            $transaction/copyright,
            utilsc:process-transaction-concept($fullDatasetTree/concept[not(@conformance='NP')])
        }
        </questionnaire>
    }
    </return>
    )
};

(:~ Central logic for updating an existing transaction representingTemplate
    @param $authmap         - required. Map derived from token
    @param $id              - required. DECOR scenario/@id to update
    @param $effectiveDate   - required. DECOR scenario/@effectiveDate to update
    @param $data            - required. DECOR scenario xml element containing everything that should be in the updated scenario
    @return concept object as xml with json:array set on elements
:)
declare function utilsc:putRepresentingTemplate($authmap as map(*), $id as xs:string, $effectiveDate as xs:string, $data as element(dataset)?, $cloneFromTrxId as xs:string?, $cloneFromTrxEd as xs:string?, $cloneFromCptId as xs:string?, $cloneFromCptEd as xs:string?, $deletelock as xs:boolean) {

    let $storedTransaction            := utillib:getTransaction($id, $effectiveDate)
    let $storedRepresentingTemplate   := $storedTransaction/representingTemplate
    let $decor                        := $storedTransaction/ancestor::decor
    let $projectPrefix                := $decor/project/@prefix
    let $defaultLanguage              := $decor/project/@defaultLanguage
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)
    
    let $locks                  := utilsc:checkScenarioObjectLock($authmap, $storedTransaction, true())
    
    let $data                   := 
        if ($data) then $data else if (empty($cloneFromTrxId)) then () else (
            let $mergeDataset  := utillib:getDatasetTree((), (), $cloneFromTrxId, $cloneFromTrxEd, true())
            return utilsc:mergeDatasets($storedTransaction, $mergeDataset, $cloneFromCptId, $cloneFromCptEd)
        )

    let $check                  :=
        if (empty($storedTransaction)) then 
            error($errors:BAD_REQUEST, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        else ()
        
    let $check                  :=
        if (count($storedTransaction) != 1) then
            error($errors:SERVER_ERROR, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' occurs ' || count($storedTransaction) || ' times. Inform your database administrator.')
        else ()
        
    let $check                  :=
        if ($storedTransaction[@type = 'group'] | $storedTransaction[transaction]) then
            error($errors:BAD_REQUEST, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' is a group. Cannot add or update a representingTemplate on a transaction group.')
        else ()
        
    let $check                  :=
        if ($storedTransaction/ancestor-or-self::*[@statusCode = $utilsc:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, 'Transaction with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it or one of its parents has one of status: ' || string-join($utilsc:STATUSCODES-FINAL, ', '))
        else ()
        
    let $check                  :=
        if (empty($cloneFromCptId) or empty($storedRepresentingTemplate)) then () 
        else if ($data[@sourceDataset = $storedRepresentingTemplate/@sourceDataset][@sourceDatasetFlexibility = $storedRepresentingTemplate/@sourceDatasetFlexibility]) then () 
        else (
            error($errors:BAD_REQUEST, 'Input SHALL match sourceDataset and sourceDatasetFlexibility of the representingTemplate to update. Found: ''' || $data/@sourceDataset || '''/''' || $data/@sourceDatasetFlexibility || ''', expected ''' || $storedRepresentingTemplate/@sourceDataset || '''/''' || $storedRepresentingTemplate/@sourceDatasetFlexibility || '''. Use patch on the transaction to update the dataset if necessary.')
        )    
    
    let $check                  := utilsc:checkConceptConstraint($data)
   
    let $preparedRepresentingTemplate   := utilsc:prepareRepresentingTemplate($data, $defaultLanguage)
    
    let $checkv                 :=
        for $question in $preparedRepresentingTemplate//enableWhen/@question
        let $conceptId          := ($question/ancestor::concept/(@id | @ref))[1]
        let $conceptName        := ($data//concept[(@id | @ref) = $conceptId]/name)[1]
        return
            if ($conceptId = $question) then 'Concept ''' || $conceptId || ''' ' ||  $conceptName || ' is enabled based on itself.'
            else if ($preparedRepresentingTemplate/concept[(@id | @ref) = $question]) then () 
            else 'Concept ''' || $conceptId || ''' ' ||  $conceptName || ' is enabled on a concept ''' || $question || ''' that is not part of the transaction.'
            
    let $check                  :=
        if (empty($checkv)) then () else (
            error($errors:BAD_REQUEST, 'Transaction concepts with enableWhen SHALL depend on one or more other concepts that are also present in the transaction. ' || normalize-space(string-join($checkv, ' ')))
        )
    
    let $check                  := ruleslib:checkDecorSchema($preparedRepresentingTemplate, 'update')
    
    (: save history:)
    let $intention              := if ($storedTransaction[@statusCode='final']) then 'patch' else 'version'
    let $history                := histlib:AddHistory($authmap?name, $decorlib:OBJECTTYPE-TRANSACTION, $projectPrefix, $intention, $storedTransaction)
    
    let $updates                := 
        if ($storedRepresentingTemplate) then update replace $storedRepresentingTemplate with $preparedRepresentingTemplate
        else update insert $preparedRepresentingTemplate into $storedTransaction
        
    let $delete                 := if ($deletelock) then update delete $locks else ()
    
    return ()
};

(:~ Central logic for patching an existing transaction statusCode, versionLabel, expirationDate and/or officialReleaseDate
    @param $authmap required. Map derived from token
    @param $id required. DECOR concept/@id to update
    @param $effectiveDate required. DECOR concept/@effectiveDate to update
    @param $recurse optional as xs:boolean. Default: false. Allows recursion into child particles to apply the same updates
    @param $listOnly optional as xs:boolean. Default: false. Allows to test what will happen before actually applying it
    @param $newStatusCode optional as xs:string. Default: empty. If empty, does not update the statusCode
    @param $newVersionLabel optional as xs:string. Default: empty. If empty, does not update the versionLabel
    @param $newExpirationDate optional as xs:string. Default: empty. If empty, does not update the expirationDate 
    @param $newOfficialReleaseDate optional as xs:string. Default: empty. If empty, does not update the officialReleaseDate 
    @return list object with success and/or error elements or error
:)
declare function utilsc:setTransactionStatus($authmap as map(*), $id as xs:string, $effectiveDate as xs:string?, $recurse as xs:boolean, $listOnly as xs:boolean, $newStatusCode as xs:string?, $newVersionLabel as xs:string?, $newExpirationDate as xs:string?, $newOfficialReleaseDate as xs:string?) as element()* {

    let $object                 := utillib:getTransaction($id, $effectiveDate)
    let $decor                  := $object/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($object) = 1) then () 
        else if ($object) then
            error($errors:SERVER_ERROR, 'Transaction id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because multiple ' || count($object) || ' were found in: ' || string-join(distinct-values($object/ancestor::decor/project/@prefix), ' / ') || '. Please inform your database administrator.')
        else (
            error($errors:BAD_REQUEST, 'Transaction id ''' || $id || ''' effectiveDate ''' || $effectiveDate || ''' cannot be updated because it cannot be found.')
        )
    
    let $check                  := utilsc:checkScenarioAccess($authmap, $decor)

    let $check                  :=
        if ($object[@statusCode = 'cancelled'] and $object[not(@statusCode = $newStatusCode)]) then
            if ($object/ancestor::scenario[@statusCode = 'draft']) then () else (
                error($errors:BAD_REQUEST, 'Status transition not allowed from ' || $object/@statusCode || ' to ''' || $newStatusCode || ''', because the scenario is not in status draft but in status ''' || $object/ancestor::scenario/@statusCode || '''')
            )
        else ()
    
    let $testUpdate             := utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, true(), $projectPrefix)
    let $results                :=
        if ($testUpdate[self::error]) then $testUpdate 
        else if ($listOnly) then $testUpdate 
        else utillib:applyStatusProperties($authmap, $object, $object/@statusCode, $newStatusCode, $newVersionLabel, $newExpirationDate, $newOfficialReleaseDate, $recurse, false(), $projectPrefix)
        
    return
        if ($results[self::error] and not($listOnly)) then
            error($errors:BAD_REQUEST, string-join(
                for $e in $results[self::error]
                return
                    $e/@itemname || ' id ''' || $e/@id || ''' effectiveDate ''' || $e/@effectiveDate || ''' cannot be updated: ' || data($e)
                , ' ')
            )
        else $results       
};

(:  Smart clone a scenario transaction representingTemplate based on a new dataset version

    Example from PRSB
    Personalised Care and Support Plan
    ----------------------------------
    Personalised Care and Support Plan has a previous scenario 2.16.840.1.113883.3.1937.777.28.4.208/2021-01-08T14:21:57
    based on an older sourceDataset.
    It shall be cloned as a new transaction but no longer based on the old dataset but on a new dataset that is a natural version
    of the old dataset. The new dataset is dataset 2.16.840.1.113883.3.1937.777.28.1.223/2024-01-24T12:52:29.

    Example from AKTIN Notaufnahmeregister
    Notaufnahmeregister has a previous scenario 2.16.840.1.113883.2.6.60.3.4.33/2019-12-05T08:17:05
    based on an older sourceDataset.
    It shall be cloned as a new transaction but no longer based on the old dataset but on a new dataset that is a natural version
    of the old dataset. The new dataset is dataset 2.16.840.1.113883.2.6.60.3.1.9/2023-11-20T00:00:00.

    So to date (Feb 2024), the call is as follows:
    let $oldscenario := utillib:getTransaction($oldscid, $oldsced)
    let $clone := local:smartUpdateRepresentingTemplate($oldscenario/representingTemplate, $newdsid, $newdsed, 'en-US')

    The (intermediate) result is returned as
    <result check="{$check}" failed="{count($reptempwithduplicates/fail)}" concepts="{count($reptemp/concept)}">
        check=number of concepts checked, fail=number of unrelated concepts where mapping failed, concepts=number of concepts in new clone
        <rawclone>...the raw sceanrio clone for inspection (debug)</rawclone>
        <representingTemplate sourceDataset="{$targetDatasetId}" sourceDatasetFlexibility="{$targetDatasetFlexibility}">
            ... the list of concepts from the old sceanrio now referencing the new dataset, 
            with all properties copied such as conditions, constarints, etc....
        </representingTemplate>
    </result>
:)
(:~ Central logic for clone a scenario transaction representingTemplate based on a new dataset version
    @param $id                      - required. DECOR transaction/@id to update the contained representingTemplate
    @param $effectiveDate           - required. DECOR transaction/@effectiveDate to update    
    @param $targetDatasetId          - required. DECOR new target dataset/@id to use as the new base
    @param $targetDatasetFlexibility - required. DECOR new target dataset/@effectiveDate to use as the new base
    @return a representingTemplate element that might be checked by the user and then be used for an update of the existing scenario transaction representingTemplate.
:)  
declare function utilsc:smartCloneRepresentingTemplate($id as xs:string, $effectiveDate as xs:string, $targetDatasetId as xs:string, $targetDatasetFlexibility as xs:string) as element() {
    
    let $storedScenario         := utillib:getTransaction($id, $effectiveDate)
    let $representingTemplate   := $storedScenario/representingTemplate
    
    let $decor                  := $storedScenario/ancestor::decor
    let $projectPrefix          := $decor/project/@prefix
    let $defaultLanguage        := ($storedScenario/ancestor::decor/project/@defaultLanguage, $storedScenario//@language, 'en-US')[1]

    let $check                  :=
        if (empty($storedScenario)) then 
            error($errors:BAD_REQUEST, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' does not exist')
        else ()
        
    let $check                  :=
        if (count($storedScenario) gt 1) then
            error($errors:SERVER_ERROR, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' occurs ' || count($storedScenario) || ' times. Inform your database administrator.')
        else ()
    
    let $check                  :=
        if ($storedScenario[@statusCode = $utilsc:STATUSCODES-FINAL]) then
            error($errors:BAD_REQUEST, 'Scenario with id ''' || $id || ''' and effectiveDate ''' || $effectiveDate || ''' cannot be updated while it one of status: ' || string-join($utilsc:STATUSCODES-FINAL, ', '))
        else ()
    
    let $targetDataset          := if ($targetDatasetId) then utillib:getDataset($targetDatasetId, $targetDatasetFlexibility) else ()
    let $check                  :=
        if ($targetDatasetId and empty($targetDataset)) then
            error($errors:BAD_REQUEST, 'RepresentingTemplate refers to non-existent new version of a dataset, id ''' || $targetDatasetId || ''' effectiveDate ''' || $targetDatasetFlexibility)
        else ()

    let $reptempwithduplicates  :=
        <representingTemplate>
        {
            for $concept in $representingTemplate/concept
            let $oref           := $concept/@ref
            let $oflx           := $concept/@flexibility
            (:
                either there is an inherit statement such as
                <inherit ref="..." flexibility="..."/>
                or there is a relationship statement such as 
                <relationship type="SPEC" ref="..." flexibility="..."/>
            :)
            let $conceptInTarget    := 
                if ($targetDataset//concept[inherit[@ref = $oref]])
                then $targetDataset//concept[inherit[@ref = $oref]]
                else $targetDataset//concept[relationship[@ref = $oref]]
            let $originalConcept    := utillib:getConcept($oref, $oflx, (), ())
            let $sourceConcept      := if ($originalConcept) then <sourceConcept>{$originalConcept/(@*, name, desc)}</sourceConcept> else ()
            
            return
                if (empty($conceptInTarget)) then 
                    element failed {
                        $concept/@*,
                        $concept/node(),
                        $sourceConcept
                    }
                else if (empty($conceptInTarget[@statusCode=$utilsc:STATUSCODES-NONDEPRECS])) then 
                    element invalid {
                        $concept/@*,
                        $concept/node(),
                        $sourceConcept
                    }
                else (
                    for $cit in $conceptInTarget[@statusCode=$utilsc:STATUSCODES-NONDEPRECS]
                    return
                        element concept {
                            $concept/(@* except (@ref|@flexibility)),
                            attribute ref { $cit/@id },
                            attribute flexibility { $cit/@effectiveDate },
                            $concept/node(),
                            $sourceConcept
                        },
                    for $xc in $conceptInTarget/ancestor::concept[@statusCode=$utilsc:STATUSCODES-NONDEPRECS]
                        let $xcid   := $xc/@id
                        let $xced   := $xc/@effectiveDate
                        return  <concept ref="{$xcid}" flexibility="{$xced}" minimumMultiplicity="0" maximumMultiplicity="*" conformance="R" added2fix="true"/>
                )
        }
        </representingTemplate>
        
    let $reptemp                :=
        <representingTemplate>
        {
            for $concept in $reptempwithduplicates/node()[name() = 'concept']
            let $cid            := $concept/@ref
            let $ced            := $concept/@flexibility
            let $cided          := concat($cid, $ced)
            group by $cided
            return if ($concept[not(@added)]) then $concept[not(@added)][1] else $concept[@added][1]
        }
        </representingTemplate>
    
   return
        <result>
        {
            attribute conceptsTransferred {count($reptemp/concept)},
            attribute added2fix {count($reptempwithduplicates/concept[@added2fix='true'])},
            attribute failed2Map {count($reptempwithduplicates/failed)},
            attribute invalidStatus {count($reptempwithduplicates/invalid)},
            attribute totalConceptsInSource {count($representingTemplate/concept)}
            ,
            <nomatches count="{count($reptempwithduplicates/failed)}">
            {
                utillib:addJsonArrayToElements($reptempwithduplicates/failed)
            }
            </nomatches>,
            <representingTemplate>
            {
                attribute sourceDataset {$targetDataset/@id},
                attribute sourceDatasetFlexibility {$targetDataset/@effectiveDate},
                $representingTemplate/(@* except (@sourceDataset|@sourceDatasetFlexibility))
                ,
                for $concept in $reptemp/node()[name() = 'concept'][not(@absent = 'true')]
                let $prsc       := utilsc:prepareRepresentingTemplateConcept($concept, $targetDataset, $defaultLanguage)
                return
                    <concept>
                    {
                        $prsc/@*,
                        for $node in $prsc/context[string-length()>0]
                        return utillib:serializeNode($node),
                        $prsc/(* except context)
                    }
                    </concept>
                    
            }
            </representingTemplate>
        }
        </result>
};

(: ===============                                               HELPER FUNCTIONS                                              ================:)

declare %private function utilsc:checkScenarioAccess($authmap as map(*), $decor as element(decor)) as element()? {
    utilsc:checkScenarioAccess($authmap, $decor, (), (), false(), false())
};

declare %private function utilsc:checkScenarioAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean) as element()? {
    utilsc:checkScenarioAccess($authmap, $decor, $id, $effectiveDate, $checkLock, false())
};

declare %private function utilsc:checkScenarioAccess($authmap as map(*), $decor as element(decor), $id as xs:string?, $effectiveDate as xs:string?, $checkLock as xs:boolean, $breakLock as xs:boolean) as element()? {

    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-SCENARIOS)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to modify scenarios/transactions in project ', $decor/project/@prefix, '. You have to be an active author in the project.'))
        )

    return
        if ($checkLock) then (
            let $lock           := decorlib:getLocks($authmap, $id, $effectiveDate, ())
            let $lock           := if ($lock) then $lock else decorlib:setLock($authmap, $id, $effectiveDate, $breakLock)
            return
                if ($lock) then $lock else (
                    error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for this scenario/transaction (anymore). Get a lock first.'))
                )    
            )
            else ()
};

(:~ Build a map that has template id+effectiveDate as key and all template meta properties (name, displayName, versionLabel, ident, url etc.) as value :)
declare %private function utilsc:buildTemplateMap($scenariosOrTransactions as element()*, $decor as element(decor)) as map(*)? {
    
    map:merge(
        for $representingTemplate in $scenariosOrTransactions/descendant-or-self::representingTemplate[@ref[not(. = '')]]
        let $tmid           := $representingTemplate/@ref
        let $tmed           := $representingTemplate/@flexibility[. castable as xs:dateTime]
        let $tmided         := concat($tmid, $tmed)
        group by $tmided
        return (
            let $tmid       := $representingTemplate[1]/@ref
            let $tmed       := $representingTemplate[1]/@flexibility[. castable as xs:dateTime]
            let $templates  :=
                if ($tmed castable as xs:dateTime) then (
                    let $t1 := $setlib:colDecorData//template[@id = $tmid][@effectiveDate = $tmed]
                    let $t2 := $setlib:colDecorCache//template[@id = $tmid][@effectiveDate = $tmed]
                    return ($t1, $t2)
                )
                else (
                    let $t1 := $setlib:colDecorData//template[@id = $tmid]
                    let $t2 := $setlib:colDecorCache//template[@id = $tmid]
                    let $tmed   := max(($t1/xs:dateTime(@effectiveDate), $t2/xs:dateTime(@effectiveDate)))
                    return ($t1, $t2)[@effectiveDate = $tmed]
                )
            return
                map:entry($tmided, <template>{$templates[1]/(@* except (@ident|@url)),
                    if ($templates[1]/ancestor::cacheme) then (
                        attribute ident {$templates[1]/ancestor::cacheme/@bbrident},
                        attribute url {$templates[1]/ancestor::cacheme/@bbrurl}
                    ) 
                    else if ($decor/project/@prefix = $templates[1]/ancestor::decor/project/@prefix) then () else (
                        attribute ident {$templates[1]/ancestor::decor/project/@prefix},
                        attribute url {utillib:getServerURLServices()}
                    )
                }
                </template>)
        )
    )
};

declare %private function utilsc:scenarioBasics($scenario as element(), $transactionmap as map(*), $projectPrefix as xs:string?, $projectLanguage as xs:string?, $templatemap as map(*)) as element() {
    
    <scenario>
    {
        attribute uuid {util:uuid()},
        $scenario/@id,
        $scenario/@effectiveDate,
        $scenario/@statusCode,
        attribute versionLabel {$scenario/@versionLabel},
        attribute expirationDate {$scenario/@expirationDate},
        attribute officialReleaseDate {$scenario/@officialReleaseDate},
        attribute canonicalUri {$scenario/@canonicalUri},
        $scenario/@lastModifiedDate,
        if ($scenario/ancestor::decor/project[@prefix = $projectPrefix]) then () else attribute ident {$scenario/ancestor::decor/project/@prefix}
        ,
        if (empty($projectLanguage)) then () 
        else if ($projectLanguage='*') then () 
        else if ($scenario/name[@language = $projectLanguage]) then () 
        else <name language="{$projectLanguage}">{($scenario/name[@language = 'en-US'], $scenario/name)[1]/node()}</name>
        ,
        $scenario/name,
        $scenario/desc,
        if ($scenario[desc/@language = $projectLanguage]) then () else <desc language="{$projectLanguage}"/>,
        $scenario/trigger,
        $scenario/condition,
        $scenario/publishingAuthority,
        $scenario/property,
        $scenario/copyright,
        for $transaction in $scenario/transaction[map:contains($transactionmap, concat(@id, @effectiveDate))]
        return utilsc:transactionBasics($transaction, $projectPrefix, $projectLanguage, $templatemap)
    }
    </scenario>
};

declare %private function utilsc:transactionBasics($transaction as element(), $projectPrefix as xs:string?, $projectLanguage as xs:string?, $templatemap as map(*)) as element() {
        
    <transaction>
    {
        attribute uuid {util:uuid()},
        $transaction/@id,
        $transaction/@effectiveDate,
        $transaction/@statusCode,
        attribute versionLabel {$transaction/@versionLabel},
        $transaction/@expirationDate,
        $transaction/@officialReleaseDate,
        attribute type {$transaction/@type},
        attribute label {$transaction/@label},
        attribute model {$transaction/@model},
        attribute canonicalUri {$transaction/@canonicalUri},
        $transaction/@lastModifiedDate,
        if ($transaction/ancestor::decor/project[@prefix = $projectPrefix]) then () else attribute ident {$transaction/ancestor::decor/project/@prefix},
        if (empty($projectLanguage)) then () 
        else if ($projectLanguage = '*') then () 
        else if ($transaction/name[@language = $projectLanguage]) then () 
        else <name language="{$projectLanguage}">{($transaction/name[@language = 'en-US'], $transaction/name)[1]/node()}</name>
        ,
        $transaction/name,
        $transaction/desc
        ,
        if (empty($projectLanguage)) then () 
        else if ($projectLanguage='*') then () 
        else if ($transaction[desc/@language=$projectLanguage]) then () 
        else <desc language="{$projectLanguage}"/>
        ,
        $transaction/trigger,
        $transaction/condition,
        $transaction/dependencies,
        $transaction/publishingAuthority,
        $transaction/property,
        $transaction/copyright
        ,
        if ($transaction/@type='group') then (
            for $t in $transaction/transaction
            return utilsc:transactionBasics($t, $projectPrefix, $projectLanguage, $templatemap)
        )
        else (
            if ($transaction[actors]) then $transaction/actors else <actors/>,
            if ($transaction[representingTemplate]) then utilsc:representingTemplateBasics($transaction/representingTemplate, $templatemap, false()) else <representingTemplate/>
        )
    }
    </transaction>
};

declare %private function utilsc:representingTemplateBasics($representingTemplate as element(representingTemplate)?, $templatemap as map(*)?, $doContents as xs:boolean) as element(representingTemplate)? {
    
    let $dataset                := if ($representingTemplate/@sourceDataset) then utillib:getDataset($representingTemplate/@sourceDataset, $representingTemplate/@sourceDatasetFlexibility) else ()
    let $template               := if ($representingTemplate[@ref]) then map:get($templatemap, concat($representingTemplate/@ref, $representingTemplate/@flexibility[. castable as xs:dateTime])) else ()
    let $questionnaire          := if ($representingTemplate/@representingQuestionnaire) then utillib:getQuestionnaire($representingTemplate/@representingQuestionnaire, $representingTemplate/@representingQuestionnaireFlexibility) else ()
    
    return
        <representingTemplate>
        {
            $representingTemplate/@sourceDataset,
            $representingTemplate/@sourceDatasetFlexibility,
            $representingTemplate/@ref,
            $representingTemplate/@flexibility,
            $representingTemplate/@representingQuestionnaire,
            $representingTemplate/@representingQuestionnaireFlexibility,
            if ($template) then (
                attribute templateId {$template/(@id|@ref)},
                attribute templateDisplayName {($template/@displayName, $template/@name)[not(. = '')][1]},
                for $att in $template/(@effectiveDate, @statusCode, @versionLabel, @expirationDate, @officialReleaseDate, @ident, @url, @canonicalUri)
                (:E.g.: attribute templateEffectiveDate {$template/@effectiveDate} :)
                return attribute {'template' || upper-case(substring(local-name($att), 1, 1)) || substring(local-name($att), 2)} {$att}
            ) else ()
            ,
            for $att in $dataset/(@effectiveDate, @statusCode, @versionLabel, @expirationDate, @officialReleaseDate, @ident, @url, @canonicalUri)
            (:E.g.: attribute sourceDatasetEffectiveDate {$dataset/@effectiveDate} :)
            return  attribute {'sourceDataset' || upper-case(substring(local-name($att), 1, 1)) || substring(local-name($att), 2)} {$att}
            ,
            for $att in $questionnaire/(@effectiveDate, @statusCode, @versionLabel, @expirationDate, @officialReleaseDate, @ident, @url, @canonicalUri)
            (:E.g.: attribute questionnaireEffectiveDate {$questionnaire/@effectiveDate} :)
            return attribute {'questionnaire' || upper-case(substring(local-name($att), 1, 1)) || substring(local-name($att), 2)} {$att}
            ,
            for $node in $dataset/name
            return <sourceDatasetName>{$node/@*, $node/node()}</sourceDatasetName>
            ,
            for $node in $questionnaire/name
            return <questionnaireName>{$node/@*, $node/node()}</questionnaireName>
            ,
            if ($doContents) then $representingTemplate/node() else ()
        }
        </representingTemplate>
};

(:~ Return the live scenarios if param decorVersion is not a dateTime OR 
    compiled, archived/released DECOR scenarios based on having any contained transaction that binds the requested dataset.
    Released/compiled matches could be present in multiple languages. Use the parameters @language to select the right one.
    
    @param id                   - required representingTemplate/@sourceDataset
    @param $effectiveDate       - optional representingTemplate/@sourceDatasetFlexibility
    @param $decorRelease        - optional decor/@versionDate
    @param $decorLanguage       - optional decor/@language. Only relevant with parameter $decorVersion. Empty uses decor/project/@defaultLanguage, '*' selects all versions, e.g. 'en-US' selects the US English version (if available)
    @return zero or more live scenarios, or 1 or more archived DECOR version instances or nothing if not found
    @since 2015-05-04
:)
declare %private function utilsc:getScenariosByDataset($id as xs:string, $effectiveDate as xs:string?, $decorPrefixOrId as xs:string?, $decorRelease as xs:string?, $decorLanguage as xs:string?) as element(scenario)* {
    
    let $decor                  := if ($decorPrefixOrId[string-length() gt 0]) then utillib:getDecor($decorPrefixOrId, $decorRelease, $decorLanguage)[1] else $setlib:colDecorData/decor
        
    let $dataset                := utillib:getDataset($id, $effectiveDate, $decorPrefixOrId, $decorRelease, $decorLanguage)
    
    for $scenario in $decor//scenario[ancestor::scenarios]
    let $datasets               :=
        for $rp in $scenario//representingTemplate[@sourceDataset = $id]
        let $ds-id  := $rp/@sourceDataset
        let $ds-ed  := $rp/@sourceDatasetFlexibility[not(. = 'dynamic')]
        group by $ds-id, $ds-ed
        return
            utillib:getDataset($ds-id, $ds-ed, $decorPrefixOrId, $decorRelease, $decorLanguage)
    return
        if ($datasets[@effectiveDate = $dataset/@effectiveDate]) then $scenario else ()

};

declare %private function utilsc:handleTransactionElements($data as element(transaction), $defaultLanguage as xs:string) as element()* {
    
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/name),
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/desc)
    ,
    for $node in $data/trigger[string-length() gt 0]
    let $node                   := if ($node[*]) then $node else utillib:parseNode($node)
    return element {name($node)} {$node/@*[not(.='')], $node/node()}
    ,
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/condition),
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/dependencies)
    ,
    for $authority in $data/publishingAuthority[@name[string-length()>0]]
    let $n                      := $authority/@name
    group by $n
    return utillib:preparePublishingAuthorityForUpdate($authority[1])
    ,
    for $node in $data/property[string-length()>0]
    let $node       := if ($node[*]) then $node else utillib:parseNode($node)
    return element {name($node)} {$node/@*[not(.='')], $node/node()}
    ,
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/copyright)
    ,
    if ($data[@type = 'group'] | $data[transaction]) then () else (
        <actors>
        {
            for $actor in $data/actors/actor[string-length(@id) gt 0]
            return <actor>{$actor/@id, $actor/@role}</actor>
        }
        </actors>
        ,
        if ($data[representingTemplate]) then
            <representingTemplate>
            {
                $data/representingTemplate/@sourceDataset[string-length() gt 0],
                $data/representingTemplate/@sourceDatasetFlexibility[string-length() gt 0],
                $data/representingTemplate/@ref[string-length() gt 0],
                $data/representingTemplate/@flexibility[string-length() gt 0],
                $data/representingTemplate/@representingQuestionnaire[string-length() gt 0],
                $data/representingTemplate/@representingQuestionnaireFlexibility[string-length() gt 0]
                ,
                let $dsid       := $data/representingTemplate/@sourceDataset[string-length() gt 0]
                let $dsed       := $data/representingTemplate/@sourceDatasetFlexibility[string-length() gt 0]
                
                return
                if ($dsid) then (
                    let $dataset    := utillib:getDataset($dsid, $dsed)
                    
                    for $concept in $data/representingTemplate/concept[not(@absent = 'true')]
                    return utilsc:prepareRepresentingTemplateConcept($concept, $dataset, $defaultLanguage)
                ) else ()
            }
            </representingTemplate>
        else ()
    )
};

declare %private function utilsc:prepareRepresentingTemplate($representingTemplate as element(), $defaultLanguage as xs:string) as element(representingTemplate) {
    
    <representingTemplate>
    {
        let $dsid               := ($representingTemplate[self::representingTemplate]/@sourceDataset,                         $representingTemplate[self::dataset]/@id)[not(. = '')][1]
        let $dsed               := ($representingTemplate[self::representingTemplate]/@sourceDatasetFlexibility,              $representingTemplate[self::dataset]/@effectiveDate)[not(. = '')][1]
        let $tmid               := ($representingTemplate[self::representingTemplate]/@ref,                                   $representingTemplate[self::dataset]/@templateId)[not(. = '')][1]
        let $tmed               := ($representingTemplate[self::representingTemplate]/@flexibility,                           $representingTemplate[self::dataset]/@templateEffectiveDate)[not(. = '')][1]
        let $qid                := ($representingTemplate[self::representingTemplate]/@representingQuestionnaire,             $representingTemplate[self::dataset]/@questionnaireId)[not(. = '')][1]
        let $qed                := ($representingTemplate[self::representingTemplate]/@representingQuestionnaireFlexibility,  $representingTemplate[self::dataset]/@questionnaireEffectiveDate)[not(. = '')][1]
        let $dataset            := if ($dsid) then utillib:getDataset($dsid, $dsed) else ()
        
        let $check              := 
            if ($dsid and empty($dataset)) then
                error($errors:BAD_REQUEST, 'RepresentingTemplate refers to non-existent dataset id ''' || $dsid || ''' effectiveDate ''' || $dsed)
            else ()
        
        return (
            if ($dsid) then (
                attribute sourceDataset {$dsid},
                if ($dsed) then attribute sourceDatasetFlexibility {$dsed} else ()
            ) else ()
            ,
            if ($tmid) then (
                attribute ref {$tmid},
                if ($tmed) then attribute flexibility {$tmed} else ()
            ) else ()
            ,
            if ($qid) then (
                attribute representingQuestionnaire {$qid},
                if ($qed) then attribute representingQuestionnaireFlexibility {$qed} else ()
            ) else ()
            ,
            for $concept in $representingTemplate/concept[not(@absent = 'true')]
            return utilsc:prepareRepresentingTemplateConcept($concept, $dataset, $defaultLanguage)
        )
    }
    </representingTemplate>
};

declare %private function utilsc:prepareRepresentingTemplateConcept($concept as element(concept), $dataset as element(dataset)?, $defaultLanguage as xs:string) as item()* {
    
    let $deid                   := ($concept/@ref, $concept/@id)[not(. = '')][1]
    let $datasetConcept         := $dataset//concept[@id = $deid]
    let $originalConcept        := utillib:getOriginalForConcept($datasetConcept[1])
    let $deed                   := ($datasetConcept/@effectiveDate, $concept/@flexibility, $concept/@effectiveDate)[not(. = '')][1]

    return (
        <concept>
        {
            attribute ref {$deid},
            if ($deed) then attribute flexibility {$deed} else ()
            ,
            switch ($concept/@conformance)
            case 'NP' return (
                attribute minimumMultiplicity {0}, 
                attribute maximumMultiplicity {0}, 
                attribute conformance {'NP'}
            )
            case 'C' return  (
                if ($concept/condition) then $concept/@conformance
                else (
                    $concept/@minimumMultiplicity[matches(., '^(\?|0|([1-9]\d*))$')], 
                    $concept/@maximumMultiplicity[matches(., '^(\?|\*|0|([1-9]\d*))$')], 
                    $concept/@conformance
                )
            )
            default return   (
                $concept/@minimumMultiplicity[matches(., '^(\?|0|([1-9]\d*))$')], 
                $concept/@maximumMultiplicity[matches(., '^(\?|\*|0|([1-9]\d*))$')], 
                $concept/@conformance[. = ('R', 'C', 'NP')]
            )
            ,
            if ($concept[@conformance = 'M']) then attribute isMandatory {'true'} else $concept/@isMandatory,
            $concept/@enableBehavior,
            comment {concat($originalConcept/@type, ' ', tokenize($deid,'\.')[last()], ' :: ', $originalConcept/name[1])}
            ,
            for $node in $concept/context[string-length()>0]
            return utillib:parseNode($node)
            ,
            if ($concept[@conformance = 'C']) then (
                for $node in $concept/condition
                return utilsc:prepareRepresentingTemplateConceptCondition($node, $defaultLanguage)
                ,
                $concept/enableWhen
                ) 
            else ()
            ,
            for $node in $concept/terminologyAssociation[@conceptId]
            return utilsc:prepareRepresentingTemplateConceptTerminologyAssociation($node)
            ,
            for $node in $concept/identifierAssociation[@conceptId]
            return utilsc:prepareRepresentingTemplateConceptIdentifierAssociation($node)
        }
        </concept>
        ,
        for $childconcept in $concept/concept[not(@absent = 'true')]
        return utilsc:prepareRepresentingTemplateConcept($childconcept, $dataset, $defaultLanguage)
    )
};

declare %private function utilsc:prepareRepresentingTemplateConceptCondition($condition as element(condition), $defaultLanguage as xs:string) as element() {
    
    <condition>
    {
        switch ($condition/@conformance)
        case 'NP' return (
            attribute minimumMultiplicity {0}, 
            attribute maximumMultiplicity {0}, 
            attribute conformance {'NP'}
        )
        case 'C' return  ((:$condition/@conformance:) (: not logically possible :))
        default return   (
            $condition/@minimumMultiplicity[matches(., '^(\?|0|([1-9]\d*))$')], 
            $condition/@maximumMultiplicity[matches(., '^(\?|\*|0|([1-9]\d*))$')], 
            $condition/@conformance[. = ('R', 'C', 'NP')]
        )
        ,
        if ($condition[@conformance = 'M']) then attribute isMandatory {'true'} else $condition/@isMandatory,
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($condition/desc)
        ,
        (: back in 'the old days' we did not have a potential multi lingual desc element under condition but just text :)
        if ($condition[*] | $condition[string-length() = 0]) then () else <desc language="{$defaultLanguage}">{data($condition)}</desc>
    }
    </condition>
};

declare %private function utilsc:prepareRepresentingTemplateConceptTerminologyAssociation($association as element(terminologyAssociation)) as element() {
    
    <terminologyAssociation>
    {
        $association/@conceptId[not(. = '')],
        $association/@conceptFlexibility[not(. = '')],
        
        (: a coded concept :)
        $association/@code[not(. = '')],
        $association/@codeSystem[not(. = '')],
        $association/@codeSystemName[not(. = '')],
        $association/@displayName[not(. = '')],
        $association/@equivalence[not(. = '')],
        
        (: a value set and flexibility :)
        $association/@valueSet[not(. = '')],
        $association/@flexibility[not(. = '')],
        $association/@strength[not(. = '')],
        
        (: version handling :)
        $association/@effectiveDate[not(. = '')],
        $association/@expirationDate[not(. = '')],
        $association/@officialReleaseDate[not(. = '')],
        $association/@versionLabel[not(. = '')]
    }
    </terminologyAssociation>
};

declare %private function utilsc:prepareRepresentingTemplateConceptIdentifierAssociation($association as element(identifierAssociation)) as element() {
    
    <identifierAssociation>
    {
        $association/@conceptId[not(. = '')],
        $association/@conceptFlexibility[not(. = '')],
        $association/@ref[not(. = '')],
        
        (: version handling :)
        $association/@effectiveDate[not(. = '')],
        $association/@expirationDate[not(. = '')],
        $association/@officialReleaseDate[not(. = '')],
        $association/@versionLabel[not(. = '')]
    }
    </identifierAssociation>
};

declare %private function utilsc:checkScenarioTransactionsForUpdate($transactions as element(transaction)*, $storedScenario as element(scenario)) {
    
    let $check                  :=
        for $transaction in $storedScenario//transactions
        let $trid               := $transaction/@id
        let $tred               := $transaction/@effectiveDate
        return
            if (count($transactions//transaction[@id = $trid][@effectiveDate = $tred])= 1) then () else (
                error($errors:BAD_REQUEST, 'Updated scenario SHALL contain at least all stored transactions for this scenario. You cannot delete transactions in an update. Missing transaction id ''' || $trid || ''' effectiveDate ''' || $tred || ''' ' || $transaction/name[1])
            )
    
    let $check                  :=
        for $transaction in $transactions[@id]
        let $trid               := $transaction/@id
        let $tred               := $transaction/@effectiveDate
        return
            if (count($storedScenario//transaction[@id = $trid][@effectiveDate = $tred])= 1) then () else (
                error($errors:BAD_REQUEST, 'Updated scenario SHALL NOT contain transaction from a different scenario. You cannot move transactions from one scenario to another in an update. Missing transaction id ''' || $trid || ''' effectiveDate ''' || $tred || ''' ' || $transaction/name[1])
            )
     return ()   
};

declare %private function utilsc:checkScenarioObjectsForUpdate($object as element(), $storedObject as element()?, $scenario as element(scenario), $decor as element(decor)) {
    
    let $storedObjectType   := upper-case(substring(local-name($object), 1, 1)) || substring(local-name($object), 2)
        
    let $check              :=
        if ($object/@id) then (
            if (empty($storedObject)) then
                error($errors:BAD_REQUEST, $storedObjectType || ' ' || $object/@id || ' effectiveDate ''' || $object/@effectiveDate || ''' SHALL exist as/in stored scenario on the server. Cannot update non-existent parts.')
            else if (count($storedObject) = 1) then () else (
                error($errors:SERVER_ERROR, $storedObjectType || ' ' || $object/@id || ' effectiveDate ''' || $object/@effectiveDate || ''' occurs ' || count($storedObject) || ''' times. Inform your database administrator.')
            ),
           
            if ($object[@statusCode = $storedObject/@statusCode]) then () else (
                error($errors:BAD_REQUEST, concat($storedObjectType || ' ' || $object/@id || ' SHALL have the same statusCode as the statusCode ''', $storedObject/@statusCode, ''' used for updating. Found in request body: ', ($scenario/@statusCode, 'null')[1]))
            )
        )
        else (
            (: we are apparently adding something :)
            if (empty($object/@statusCode)) then () else if ($object[@statusCode = $utilsc:ITEM-STATUS]) then () else (
                error($errors:BAD_REQUEST, $storedObjectType || ' ' || $object/@statusCode || ' SHALL be one of ', string-join($utilsc:ITEM-STATUS, ', '))
            )
        )
        
      let $check              :=  
        if ($object[not(name)]) then
            error($errors:BAD_REQUEST, $storedObjectType || ' ' || $object/@id || ' SHALL have at least one name')
        else ()
      
      let $check              := if ($object[self::transaction]) then ruleslib:checkTransaction($object, $decor, 'update') else ()
    
    return ()
};

declare %private function utilsc:checkScenarioObjectLock($authmap as map(*), $object as element(), $recurse as xs:boolean) as element()* {

    let $lock                   := decorlib:getLocks($authmap, $object, $recurse)
    let $lock                   := if ($lock) then $lock else decorlib:setLock($authmap, $object, false())
    return
        if ($lock) then $lock else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have a lock for ' || local-name($object) || ' ' || $object/@id || ' (anymore). Get a lock first.'))
        )    
};

declare %private function utilsc:prepareScenarioForUpdate($scenario as element(), $storedScenario as element()?, $baseIdTransaction as xs:string) as element() {
    
    let $editedScenario   := if ($scenario[edit]) then $scenario else $storedScenario
    
    return
    <scenario>
    {
        if ($storedScenario/@id) then $storedScenario/@id else $scenario/@id,
        if ($storedScenario/@effectiveDate[. castable as xs:dateTime]) then $storedScenario/@effectiveDate 
        else if ($scenario/@effectiveDate[. castable as xs:dateTime]) then $scenario/@effectiveDate
        else attribute effectiveDate {substring(string(current-dateTime()),1,19)}
        ,
        if ($storedScenario/@statusCode) then $storedScenario/@statusCode
        else if ($scenario/@statusCode) then $scenario/@statusCode
        else attribute statusCode {'draft'}
        ,
        $editedScenario/@versionLabel[not(.='')],
        $editedScenario/@expirationDate[not(.='')],
        $editedScenario/@officialReleaseDate[not(.='')],
        $editedScenario/@canonicalUri[not(.='')],
        $editedScenario/@lastModifiedDate[not(.='')],
        utilsc:handleScenarioElements($editedScenario, ($storedScenario/ancestor::decor/project/@defaultLanguage, $storedScenario/name/@language, $scenario/name/@language, 'en-US')[1], false())
        ,
        for $transaction in $editedScenario/transaction
        return utilsc:prepareTransactionForUpdate($transaction, $storedScenario, $baseIdTransaction)
    }
    </scenario>
};

declare %private function utilsc:handleScenarioElements($data as element(), $defaultLanguage as xs:string, $clone as xs:boolean) as element()* {

    if ($data/name[@language = $defaultLanguage]) then () else (
        <name language="{$defaultLanguage}">
        {
            data($data/name[1]),
            if ($clone) then '(2)' else()
        }
        </name>
    ),
    
    for $lang in distinct-values($data/name/@language)
    let $node     := $data/name[@language = $lang][1]
    let $node     := if ($node[*]) then $node else utillib:parseNode($node)
    return element {name($node)} {$node/@*[not(.='')], data($node), if ($clone) then '(2)' else()}
    ,
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/desc),
    utillib:prepareTransactionTriggerForUpdate($data/trigger),
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/condition),
    utillib:preparePublishingAuthorityForUpdate($data/publishingAuthority),
    utillib:prepareConceptPropertyForUpdate($data/property),
    utillib:prepareFreeFormMarkupWithLanguageForUpdate($data/copyright)
};

declare %private function utilsc:prepareTransactionForUpdate($transaction as element(), $storedScenario as element()?, $baseIdTransaction as xs:string) as element(transaction) {
    
    let $storedTransaction  := if ($transaction[string-length(@effectiveDate) gt 0]) 
        then $storedScenario//transaction[@id = $transaction/@id][@effectiveDate = $transaction/@effectiveDate] 
        else $storedScenario//transaction[@id = $transaction/@id]
        
    let $editedTransaction  := if ($transaction[edit]) then $transaction else $storedTransaction
    let $defaultLanguage    := ($storedScenario/ancestor::decor/project/@defaultLanguage, $storedScenario//@language, 'en-US')[1]
    
    let $transactionType    :=
        if ($editedTransaction[transaction]) then attribute type {'group'} 
        else if ($editedTransaction/@type[not(. = 'group')]) then $editedTransaction/@type 
        else if ($storedTransaction/@type[not(. = 'group')]) then $storedTransaction/@type 
        else attribute type {'stationary'}
        
    return
    <transaction>
    {
        if ($storedTransaction/@id) then $storedTransaction/@id 
        else if ($transaction/@id) then $transaction/@id
        else (
            let $tridnext   := data($storedScenario/ancestor::scenarios[1]/@nextTransactionLeaf)
            let $update     := update value $storedScenario/ancestor::scenarios[1]/@nextTransactionLeaf with xs:integer($tridnext) + 1
            return attribute id {$baseIdTransaction || '.' || $tridnext}
        ),
        if ($storedTransaction[@effectiveDate castable as xs:dateTime]) then $storedTransaction/@effectiveDate
        else if ($transaction/@effectiveDate[. castable as xs:dateTime]) then $transaction/@effectiveDate
        else if ($storedTransaction/ancestor::*[@effectiveDate castable as xs:dateTime]) then ($storedTransaction/ancestor::*/@effectiveDate[. castable as xs:dateTime])[last()]
        else attribute effectiveDate {substring(string(current-dateTime()),1,19)}
        ,
        if ($storedTransaction/@statusCode) then $storedTransaction/@statusCode
        else if ($transaction/@statusCode) then $transaction/@statusCode
        else attribute statusCode {'draft'}
        ,       
        $editedTransaction/@versionLabel[not(.='')],
        $editedTransaction/@expirationDate[not(.='')],
        $editedTransaction/@officialReleaseDate[not(.='')],
        $transactionType,
        $editedTransaction/@label[not(.='')],
        $editedTransaction/@model[not(.='')],
        $editedTransaction/@canonicalUri[not(.='')],
        $editedTransaction/@lastModifiedDate[not(.='')],
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTransaction/name),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTransaction/desc)
        ,
        for $trigger in $editedTransaction/trigger[string-length()>0]
        let $trigger    := utillib:parseNode($trigger)
        return <trigger>{$trigger/@*[not(.='')], $trigger/node()}</trigger>
        ,
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTransaction/condition),
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTransaction/dependencies)
        ,
        
        for $authority in $editedTransaction/publishingAuthority[@name[string-length()>0]]
        let $n := $authority/@name
        group by $n
        return utillib:preparePublishingAuthorityForUpdate($authority[1])
        ,
        for $node in $editedTransaction/property[string-length()>0]
        return utillib:parseNode($node)
        ,
        utillib:prepareFreeFormMarkupWithLanguageForUpdate($editedTransaction/copyright)
        ,
        if ($transactionType = 'group') then
            for $t in $transaction/transaction
            return utilsc:prepareTransactionForUpdate($t, $storedScenario, $baseIdTransaction)
        else (
            <actors>
            {
                for $actor in $editedTransaction/actors/actor[not(string(@id) = '')][not(string(@role) = '')]
                return <actor>{$actor/@id, $actor/@role}</actor>
            }
            </actors>
            ,
            if ($editedTransaction/representingTemplate/(@ref|@sourceDataset|@representingQuestionnaire)[string-length()>0]) then (
                <representingTemplate>
                {
                    $editedTransaction/representingTemplate/@sourceDataset[string-length() gt 0],
                    $editedTransaction/representingTemplate/@sourceDatasetFlexibility[string-length() gt 0],
                    $editedTransaction/representingTemplate/@ref[string-length() gt 0],
                    $editedTransaction/representingTemplate/@flexibility[string-length() gt 0],
                    $editedTransaction/representingTemplate/@representingQuestionnaire[string-length() gt 0],
                    $editedTransaction/representingTemplate/@representingQuestionnaireFlexibility[string-length() gt 0],
                    (:merge with stored transaction concepts as these would not be part of a save in the 
                    scenario editor, but would come from the transaction editor. suppose that you edit a 
                    scenario before refreshing the scenario editor, you might actually overwrite what you 
                    have done in the transaction editor. :)
                    if ($storedTransaction/representingTemplate[@sourceDataset = $editedTransaction/representingTemplate/@sourceDataset]) then (
                        let $dataset    := utillib:getDataset($storedTransaction/representingTemplate/@sourceDataset[string-length() gt 0], $storedTransaction/representingTemplate/@sourceDatasetFlexibility[string-length() gt 0])
                        
                        for $concept in $storedTransaction/representingTemplate[@sourceDataset[string-length() gt 0]]/concept[not(@absent = 'true')]
                        return utilsc:prepareRepresentingTemplateConcept($concept, $dataset, $defaultLanguage)
                    )
                    else ()
                }
                </representingTemplate>
            ) else ()
        )
    }
    </transaction>
};

declare %private function utilsc:checkScenarioParameters($parameters as element(parameters), $scenario as element(scenario)*) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    let $checkn                 := (
        if ($parameters/parameter[@path = '/name']) then
            if (count($scenario/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A scenario SHALL have at least one name. You cannot remove every name.'
            )
        else ()
    )
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op             := $param/@op
        let $path           := $param/@path
        let $value          := $param/@value
        let $pathpart       := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($scenario, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($scenario/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove' or ($op = 'replace' and string($value) = '')) then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/versionLabel' return (
                if ($op = 'remove') then () else
                (:if ($op = 'remove' or ($op = 'replace' and string($value) = '')) then () else:) 
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/name' 
            case '/desc' 
            case '/trigger'
            case '/condition'
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/property' return (
                if ($param[count(value/property) = 1]) then ruleslib:checkProperty($param/value/property, $op, $path) else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/property) 
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), ' '))
        )
};

declare %private function utilsc:patchScenarioParameters($parameters as element(parameters), $scenario as element(scenario)*) {

for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/statusCode' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $scenario/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/expirationDate'
        case '/officialReleaseDate'
        case '/canonicalUri'
        case '/versionLabel' return (
            let $attname        := substring-after($param/@path, '/')
            let $op             := if ($param[string-length(@value) = 0]) then 'remove' else $param/@op
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $scenario/@*[name() = $attname]
            
            return
            switch ($op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/name' 
        case '/desc'
        case '/condition' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored         := $scenario/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/trigger' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareTransactionTriggerForUpdate($param/value/*[name() = $elmname])
            let $stored         := $scenario/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/publishingAuthority' return (
            (: only one possible :)
            let $new            := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
            let $stored         := $scenario/publishingAuthority
            
            return
            switch ($param/@op)
            case ('add') return update insert $new into $scenario
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/property' return (
            (: multiple possible per language :)
            let $new            := utillib:prepareConceptPropertyForUpdate($param/value/property)
            let $stored         := $scenario/property[@name = $new/@name][lower-case(normalize-space(string-join(.//text(), ''))) = lower-case(normalize-space(string-join($new//text(), '')))]
            
            return
            switch ($param/@op)
            case ('add') return update insert $new into $scenario
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/copyright' return (
            (: only one per language :)
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
            let $stored         := $scenario/copyright[@language = $param/value/copyright/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $scenario
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};

declare %private function utilsc:handleScenario($scenario as element(scenario)) as element(scenario) {
    
    <scenario>
    {
        $scenario/@id,
        $scenario/@effectiveDate,
        $scenario/@statusCode,
        $scenario/@versionLabel[not(. = '')],
        $scenario/@expirationDate[not(. = '')],
        $scenario/@officialReleaseDate[not(. = '')],
        $scenario/@canonicalUri[not(. = '')],
        $scenario/@lastModifiedDate[not(.='')],
        $scenario/name,
        $scenario/desc,
        $scenario/trigger,
        $scenario/condition,
        $scenario/publishingAuthority,
        $scenario/property,
        $scenario/copyright,
        for $tr in $scenario/transaction
        return utilsc:handleTransaction($tr)
    }
    </scenario>
};

declare %private function utilsc:handleTransaction($transaction as element(transaction)) as element(transaction) {
    
    <transaction>
    {
        $transaction/@id,
        $transaction/@effectiveDate,
        $transaction/@statusCode,
        $transaction/@versionLabel[not(. = '')],
        $transaction/@expirationDate[not(. = '')],
        $transaction/@officialReleaseDate[not(. = '')],
        $transaction/@type[not(. = '')],
        $transaction/@label[not(. = '')],
        $transaction/@model[not(. = '')],
        $transaction/@canonicalUri[not(. = '')],
        $transaction/@lastModifiedDate[not(.='')],
        $transaction/name,
        $transaction/desc,
        $transaction/trigger,
        $transaction/condition,
        $transaction/dependencies,
        $transaction/publishingAuthority,
        $transaction/property,
        $transaction/copyright,
        if ($transaction/transaction) then
            for $tr in $transaction/transaction
            return utilsc:handleTransaction($tr)
        else (
            $transaction/actors,
            $transaction/representingTemplate
        )
    }
    </transaction>
};

declare %private function utilsc:checkScenarioActorParameters($parameters as element(parameters), $actor as element()*) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    let $checkn                 := (
        if ($parameters/parameter[@path = '/name']) then
            if (count($actor/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'An actor SHALL have at least one name. You cannot remove every name.'
            )
        else ()
        
    )
    let $check                  :=
        for $param in $parameters/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/type' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($param[@value = $utilsc:ACTOR-TYPE/enumeration/@value]) then () else (
                    concat('Parameter value ''', $param/@value, ''' for type not supported. Supported are: ', string-join($utilsc:ACTOR-TYPE/enumeration/@value, ' '))
                )
            )
            case '/name' 
            case '/desc' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path) else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            default return ( 
                concat('Parameter combination not supported: op=''', $param/@op, ''' path=''', $param/@path, '''', if ($param[@value]) then concat(' value=''', $param/@value, '''') else string-join($param/value/*/name(), ' '), '.')
            )
     
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), ' '))
        )
};

declare %private function utilsc:patchScenarioActorParameters($parameters as element(parameters), $actor as element()*) {

    for $param in $parameters/parameter
    return
        switch ($param/@path)
        case '/type' return (
            let $attname        := substring-after($param/@path, '/')
            let $new            := attribute {$attname} {$param/@value}
            let $stored         := $actor/@*[name() = $attname]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $actor
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        case '/name' 
        case '/desc' return (
            (: only one per language :)
            let $elmname        := substring-after($param/@path, '/')
            let $new            := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
            let $stored         := $actor/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
            
            return
            switch ($param/@op)
            case ('add') (: fall through :)
            case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $actor
            case ('remove') return update delete $stored
            default return ( (: unknown op :) )
        )
        default return ( (: unknown path :) )
};

declare %private function utilsc:checkTransactionParameters($parameters as element(parameters), $transaction as element(transaction)*, $decor as element(decor)) {

    let $check                  :=
        if ($parameters[parameter]) then () else (
            error($errors:BAD_REQUEST, 'Submitted data shall be an array of ''parameter''.')
        )
    
    let $unsupportedops         := $parameters/parameter[not(@op = ('add', 'replace', 'remove'))]
    let $check                  :=
        if ($unsupportedops) then
            error($errors:BAD_REQUEST, 'Submitted parameters shall have supported ''op'' value. Found ''' || string-join(distinct-values($unsupportedops/@op), ''', ''') || ''', expected ''add'', ''replace'', or ''remove''') 
        else ()
    
    let $checkn                 := (
        if ($parameters/parameter[@path = '/name']) then
            if (count($transaction/name) - count($parameters/parameter[@op = 'remove'][@path = '/name']) + count($parameters/parameter[@op = ('add', 'replace')][@path = '/name']) ge 1) then () else (
                'A transaction SHALL have at least one name. You cannot remove every name.'
            )
        else ()
    )
    
    let $check                  :=
        for $param in $parameters/parameter
        let $op         := $param/@op
        let $path       := $param/@path
        let $value      := $param/@value
        let $pathpart   := substring-after($path, '/')
        return
            switch ($path)
            case '/statusCode' return (
                if ($op = 'remove') then
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if (utillib:isStatusChangeAllowable($transaction, $value)) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Supported are: ' || string-join(map:get($utillib:itemstatusmap, string($transaction/@statusCode)), ', ')
                )
            )
            case '/expirationDate'
            case '/officialReleaseDate' return (
                if ($op = 'remove' or ($op = 'replace' and string($value) = '')) then () else 
                if ($value castable as xs:dateTime) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL be yyyy-mm-ddThh:mm:ss.'
                )
            )
            case '/canonicalUri'
            case '/model'
            case '/label'
            case '/versionLabel' return (
                if ($op = 'remove') then () else
                if ($value[not(. = '')]) then () else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' with value ''' || $value || '''. Value SHALL NOT be empty.'
                )
            )
            case '/type' return (
                if ($op = 'remove') then 
                    'Parameter path ''' || $path || ''' does not support ' || $op
                else 
                if ($value = $utilsc:TRANSACTION-TYPE/enumeration/@value) then 
                    switch ($value)
                    case 'group' return 
                        if ($transaction[transaction] | $transaction[empty(representingTemplate)]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Type has value ''group'' but the target transaction is not a group.'
                        )
                    default return 
                        if ($transaction[not(@type = 'group')]) then () else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Type has value ''' || $value || ''' but the target transaction is a group.'
                        )
                else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || '''. Type has unsupported value ' || $value || '. SHALL be one of: ' || string-join($utilsc:TRANSACTION-TYPE/enumeration/@value, ', ')
                )
            )
            case '/name' 
            case '/desc' 
            case '/trigger'
            case '/condition'
            case '/dependencies'
            case '/copyright' return (
                if ($param[count(value/*[name() = $pathpart]) = 1]) then ruleslib:checkFreeFormMarkupWithLanguage($param/value/*[name() = $pathpart], $op, $path) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/*[name() = $pathpart]) 
                )
            )
            case '/publishingAuthority' return (
                if ($param[count(value/publishingAuthority) = 1]) then ruleslib:checkPublishingAuthority($param/value/publishingAuthority, $op, $path) else (
                    'Parameter ' || $op || ' not allowed for ''' || $path || ''' SHALL have a single publishingAuthority under value. Found ' || count($param/value/publishingAuthority)
                )
            )
            case '/property' return (
                if ($param[count(value/property) = 1]) then ruleslib:checkProperty($param/value/property, $op, $path) else (
                    'Parameter ' || $op || ' of path ' || $path || ' not supported. Input SHALL have exactly one ' || $pathpart || ' under value. Found ' || count($param/value/property) 
                )
            )
            case '/actors' return (
                if ($op = 'remove') then
                    if ($transaction[@type = 'group']) then () else (
                        'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type other than group SHALL have actors containing everything that needs to be present after patching the transaction.' 
                    )
                else 
                if ($transaction[@type = 'group']) then 
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type group SHALL NOT have actors.' 
                else 
                if ($param[count(value/actors) = 1]) then
                    for $actor in $param/value/actors/actor
                    return
                        if ($actor[@id = $decor/scenarios/actors/actor/@id]) then
                            if ($actor[@role = $utilsc:ACTOR-ROLE/enumeration/@value]) then () else (
                                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Actor has unsupported role ''' || $actor/@role || '''. Supported are: ' || string-join($utilsc:ACTOR-ROLE/enumeration/@value, ', ')
                            )
                        else (
                            'Parameter ' || $op || ' not allowed for ''' || $path || '''. Actor with id ''' || $actor/@id || ''' does not exist in the project yet.' 
                        )
                else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one actors under value. Found ' || count($param/value/actors) 
                )
            )
            case '/representingTemplate' return (
                if ($op != 'replace') then (
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A patch on the entire transaction representingTemplate can only replace.' 
                )
                else if ($transaction[@type = 'group']) then 
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type group SHALL NOT have representingTemplate.' 
                else if ($param[count(value/representingTemplate) = 1]) then (
                    if ($param/value/representingTemplate[@sourceDataset[not(. = '')]]) then () else (
                        'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction dataset reference SHALL have a sourceDataset.'
                    )
                ) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one representingTemplate under value. Found ' || count($param/value/representingTemplate) 
                )
            )
            case '/representingTemplate/dataset' return (
                if ($op = 'remove') then ()
                else if ($transaction[@type = 'group']) then 
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type group SHALL NOT have representingTemplate.' 
                else if ($param[count(value/representingTemplate) = 1]) then (
                    if ($param/value/representingTemplate[@sourceDataset[not(. = '')]]) then () else (
                        'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction dataset reference SHALL have a sourceDataset.'
                    )
                ) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one representingTemplate under value. Found ' || count($param/value/representingTemplate) 
                )
            )
            case '/representingTemplate/template' return (
                if ($op = 'remove') then ()
                else if ($transaction[@type = 'group']) then 
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type group SHALL NOT have representingTemplate.' 
                else if ($param[count(value/representingTemplate) = 1]) then (
                    if ($param/value/representingTemplate[@ref[not(. = '')]]) then () else (
                        'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction template reference SHALL have a ref.'
                    )
                ) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one representingTemplate under value. Found ' || count($param/value/representingTemplate) 
                )
            )
            case '/representingTemplate/questionnaire' return (
                if ($op = 'remove') then ()
                else if ($transaction[@type = 'group']) then 
                    'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction of type group SHALL NOT have representingTemplate.' 
                else if ($param[count(value/representingTemplate) = 1]) then (
                    if ($param/value/representingTemplate[@representingQuestionnaire[not(. = '')]]) then () else (
                        'Parameter path ''' || $path || ''' does not support ' || $op || '. A transaction template reference SHALL have a representingQuestionnaire.'
                    )
                ) else (
                    'Parameter ' || $param/@op || ' of path ' || $param/@path || ' not supported. Input SHALL have exactly one representingTemplate under value. Found ' || count($param/value/representingTemplate) 
                )
            )
            default return (
                'Parameter ' || $op || ' not allowed for ''' || $path || '''. Path value not supported'
            )
     
     return
        if (empty($checkn) and empty($check)) then () else (
            error($errors:BAD_REQUEST,  string-join (($checkn, $check), ' '))
        )
};

declare %private function utilsc:patchTransactionParameters($parameters as element(parameters), $transaction as element(transaction)*) {

        for $param in $parameters/parameter
        return
            switch ($param/@path)
            case '/statusCode'
            case '/type' return (
                let $attname  := substring-after($param/@path, '/')
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $transaction/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/expirationDate'
            case '/officialReleaseDate'
            case '/canonicalUri'
            case '/model'
            case '/label'
            case '/versionLabel' return (
                let $attname  := substring-after($param/@path, '/')
                let $op       := if ($param[string-length(@value) = 0]) then 'remove' else $param/@op
                let $new      := attribute {$attname} {$param/@value}
                let $stored   := $transaction/@*[name() = $attname]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/name' 
            case '/desc'
            case '/trigger'
            case '/condition' 
            case '/dependencies' return (
                (: only one per language :)
                let $elmname  := substring-after($param/@path, '/')
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/*[name() = $elmname])
                let $stored   := $transaction/*[name() = $elmname][@language = $param/value/*[name() = $elmname]/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/publishingAuthority' return (
                (: only one possible :)
                let $new      := utillib:preparePublishingAuthorityForUpdate($param/value/publishingAuthority)
                let $stored   := $transaction/publishingAuthority
                
                return
                switch ($param/@op)
                case ('add') return update insert $new into $transaction
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/property' return (
                (: multiple possible per language :)
                let $new      := utillib:prepareConceptPropertyForUpdate($param/value/property)
                let $stored   := $transaction/property[@name = $new/@name][lower-case(normalize-space(string-join(.//text(), ''))) = lower-case(normalize-space(string-join($new//text(), '')))]
                
                return
                switch ($param/@op)
                case ('add') return update insert $new into $transaction
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/copyright' return (
                (: only one per language :)
                let $new      := utillib:prepareFreeFormMarkupWithLanguageForUpdate($param/value/copyright)
                let $stored   := $transaction/copyright[@language = $param/value/copyright/@language]
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/actors' return (
                let $new      := utilsc:handleTransactionActors($param/value/actors)
                let $stored   := $transaction/actors
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return if ($stored) then update replace $stored with $new else update insert $new into $transaction
                case ('remove') return update delete $stored
                default return ( (: unknown op :) )
            )
            case '/representingTemplate' return (
                let $stored := $transaction/representingTemplate
                let $new    := $param/value/representingTemplate
                (: op can only be replace :)
                return
                if ($stored and $new) then update replace $stored with $new else ()
            )
            case '/representingTemplate/dataset' return (
                let $dsid   := $param/value/representingTemplate/@sourceDataset
                let $dsed   := $param/value/representingTemplate/@sourceDatasetFlexibility[. castable as xs:dateTime or . = 'dynamic']
                let $stored := $transaction/representingTemplate
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    if ($stored) then (
                        (: if we're changing datasets then best delete concepts from the old one :)
                        if ($stored[@sourceDataset = $dsid][@sourceDatasetFlexibility = $dsed]) then () else update delete $stored/node(),
                        if ($stored[@sourceDataset]) then update value $stored/@sourceDataset with $dsid else update insert $dsid into $stored,
                        if ($dsed) then 
                            if ($stored[@sourceDatasetFlexibility]) then update value $stored/@sourceDatasetFlexibility with $dsed else update insert $dsed into $stored
                        else (
                            update delete $stored/@sourceDatasetFlexibility
                        )
                    )
                    else (
                        update insert <representingTemplate>{$dsid, $dsed}</representingTemplate> into $transaction
                    )
                )
                case ('remove') return (
                    update delete $stored/@sourceDataset,
                    update delete $stored/@sourceDatasetFlexibility,
                    (: if we're deleting the source dataset then best delete any concepts :)
                    update delete $stored/concept
                )
                default return ( (: unknown op :) )
            )
            case '/representingTemplate/template' return (
                let $tmid   := $param/value/representingTemplate/@ref
                let $tmed   := $param/value/representingTemplate/@flexibility[. castable as xs:dateTime or . = 'dynamic']
                let $stored := $transaction/representingTemplate
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    if ($stored) then (
                        if ($stored[@ref]) then update value $stored/@ref with $tmid else update insert $tmid into $stored,
                        if ($tmed) then 
                            if ($stored[@flexibility]) then update value $stored/@flexibility with $tmed else update insert $tmed into $stored
                        else (
                            update delete $stored/@flexibility
                        )
                    )
                    else (
                        update insert <representingTemplate>{$tmid, $tmed}</representingTemplate> into $transaction
                    )
                )
                case ('remove') return (
                    update delete $stored/@ref,
                    update delete $stored/@flexibility
                )
                default return ( (: unknown op :) )
            )
            case '/representingTemplate/questionnaire' return (
                let $qid   := $param/value/representingTemplate/@representingQuestionnaire
                let $qed   := $param/value/representingTemplate/@representingQuestionnaireFlexibility[. castable as xs:dateTime or . = 'dynamic']
                let $stored := $transaction/representingTemplate
                
                return
                switch ($param/@op)
                case ('add') (: fall through :)
                case ('replace') return (
                    if ($stored) then (
                        if ($stored[@representingQuestionnaire]) then update value $stored/@representingQuestionnaire with $qid else update insert $qid into $stored,
                        if ($qed) then 
                            if ($stored[@representingQuestionnaireFlexibility]) then update value $stored/@representingQuestionnaireFlexibility with $qed else update insert $qed into $stored
                        else (
                            update delete $stored/@representingQuestionnaireFlexibility
                        )
                    )
                    else (
                        update insert <representingTemplate>{$qid, $qed}</representingTemplate> into $transaction
                    )
                )
                case ('remove') return (
                    update delete $stored/@representingQuestionnaire,
                    update delete $stored/@representingQuestionnaireFlexibility
                )
                default return ( (: unknown op :) )
            )
            default return ( (: unknown path :) )

};

declare %private function utilsc:handleTransactionActors($actors as element(actors)) as element(actors) {
    <actors>
    {
        for $actor in $actors/actor[string-length(@id) gt 0] 
        return <actor>{$actor/@id, $actor/@role[not(. = '')]}</actor>
    }
    </actors>
};

declare %private function utilsc:process-transaction-concept($concepts as element(concept)*) as element(item)* {
    
    for $concept in $concepts
    let $defaultValue           := ($concept/valueDomain/property/@default)[1]
    let $operhasqm              := contains(string-join($concept/operationalization, ' '), '?')
    return
        <item>
        {
            (: NB keep value of @linkId in sync with scapi:process-questionnairetransform-request#2 for @elementId :)
            attribute linkId {$concept/@id},
            attribute type {utilsc:decor2questionnaireType($concept)},
            attribute required {if ($concept[@conformance = 'NP'] | $concept[@maximumMultiplicity = '0']) then false() else exists($concept/@minimumMultiplicity[not(. = '0')])},
            attribute repeats { if ($concept[@conformance = 'NP'] | $concept[@maximumMultiplicity = ('0', '1')]) then false() else true()},
            attribute readOnly {exists($concept/valueDomain/property[@fixed = 'true'])},
            if (count($concept/valueDomain/property) = 1) then (
                $concept/valueDomain/property/@minLength[. castable as xs:integer],
                $concept/valueDomain/property/@maxLength[. castable as xs:integer],
                if ($concept/valueDomain/property/@minInclude) then attribute minValue {$concept/valueDomain/property/@minInclude} else (),
                if ($concept/valueDomain/property/@maxInclude) then attribute maxValue {$concept/valueDomain/property/@maxInclude} else ()
            )
            else (),
            if ($concept/valueDomain/property/@fractionDigits) then
                attribute maxDecimalPlaces { 
                    max(
                        for $t in $concept/valueDomain/property/@fractionDigits
                        let $md   := replace($t, '[^\d]', '')[not(. = '')]
                        return if ($md castable as xs:integer) then xs:integer($md) else ()
                    )
                }
            else (),
            $concept/@enableBehavior,
            attribute hiddenItem {false()}
            ,
            (:
                text
                to get Questionnaire item text populated: if there is an operationalization in the concept
                then this operationalization text becomes the Questionnaire item text
                otherwise the Questionnaire item text is the name of the concept
                
                see https://art-decor.atlassian.net/browse/AD30-1379 and https://art-decor.atlassian.net/browse/AD30-1642
                
                so modifed 2024-07-30
                use operationalization in data elements / scenarios for Questionnaire item text only, if the text contains a question mark
                if operationalization does not contain a question mark simply put it down as help text candidate
            :)
            let $itemText   := if ($operhasqm) then $concept/operationalization else $concept/name
            
            for $text in $itemText
            return
                <text>
                {
                    $text/@*,
                    $text/node()
                }
                </text>
            ,
            (: 
                if operationalization contains a question mark keep the original concept name(s) as synonym, 
                otherwise keep the concept operationalization
                see https://art-decor.atlassian.net/browse/AD30-1642
            :)
            if ($operhasqm) then
                for $cn in $concept/name
                return
                    <synonym>
                    {
                        $cn/@*,
                        $cn/node()
                    }
                    </synonym>
                else $concept/operationalization
            ,
            (:code:)
            for $terminologyAssociation in $concept/terminologyAssociation[@conceptId = ($concept/@id | $concept/inherit/@ref)][@code]
            return <code>{$terminologyAssociation/(@code, @codeSystem, @displayName, @canonicalUri)}</code>
            ,
            for $enableWhen in $concept/enableWhen
            return
                <enableWhen>
                {
                    for $attr in $enableWhen/@*[not(. = '')]
                    return
                        switch (local-name($attr))
                        case 'question' return (
                            if (utillib:isOid($attr)) then $attr
                            else if (utillib:isOid(substring-before($attr, '--'))) then attribute question {substring-before($attr, '--')}
                            (: we really tried but don't know what this links to :)
                            else $attr
                        )
                        default return $attr
                    ,
                    $enableWhen/node()
                }
                </enableWhen>
            ,
            (: http://hl7.org/fhir/questionnaire.html: que-4: A question cannot have both answerOption and answerValueSet:)
            (:answerValueSet:)
            for $vs in $concept/valueSet
                let $flexibility    :=
                    if ($vs/terminologyAssociation/@flexibility = 'dynamic') then ('dynamic') else $vs/@effectiveDate 
                let $canonicalUri   := 
                     if ($vs[@canonicalUri]) 
                        then $vs/@canonicalUri 
                        else if ($flexibility = 'dynamic') then concat(utillib:getServerURLFhirCanonicalBase(),'ValueSet/', $vs/@id)
                        else concat(utillib:getServerURLFhirCanonicalBase(),'ValueSet/', $vs/@id, '--', replace($flexibility,'[^\d]',''))
                
                return    
                <answerValueSet>
                {
                    attribute ref { $vs/@id },
                    attribute flexibility { $flexibility },
                    attribute canonicalUri { $canonicalUri },
                    $vs/@name,
                    $vs/@displayName,
                    $vs/@statusCode,
                    $vs/@strength
                }    
                </answerValueSet>
                
            ,
            (:answerOptions:)
            (:uncoded answerOptions:)
            for $c in $concept[empty(valueSet)]/valueDomain[@type = ('code', 'ordinal')]/conceptList/(concept | exception)
            let $tas    := $c/ancestor::concept[1]/terminologyAssociation[@conceptId = $c/@id]
            return
                <answerOption>
                {
                    $c/@ordinal[not(. = '')],
                    if ($tas) then
                        for $ta in $tas
                        return
                            <valueCoding>
                            {
                                $ta/@code[not(. = '')],
                                $ta/@codeSystem[not(. = '')],
                                $ta/@codeSystemName[not(. = '')],
                                $ta/@displayName[not(. = '')],
                                $ta/@canonicalUri[not(. = '')]
                            }
                            </valueCoding>
                    else (
                        for $cn in $c/name
                        return
                            <valueString>
                            {
                                $cn/@language,
                                attribute value { $cn/text() }
                            }
                            </valueString>
                    )
                }
                </answerOption>
            ,
            if ($defaultValue) then 
                (: For code, Questionnaire expect Coding, DECOR default is just a simple string :)
                if ($concept/*:valueDomain/@type != 'code') then
                    <initial>
                    {
                        element {utilsc:decor2questionnaireAnswerType($concept)} {
                            attribute value {$defaultValue}
                        }
                    }
                    </initial>
                else ()
            else ()
            ,
            utilsc:process-transaction-concept($concept/concept[not(@conformance='NP')])
        }
        </item>
};

declare %private function utilsc:decor2questionnaireType($concept as element(concept)) as xs:string {
    
    let $valueDomainType        := $concept/valueDomain/@type
    let $conceptItemType        := $concept/property[@name = 'Quest::ItemType']
    let $conceptItem            := $conceptItemType/normalize-space(string-join(.//text()))
    
    return
    (: allow override of all other options by explicitly setting the questionnaire item type value.
        https://hl7.org/fhir/R4/valueset-item-type.html
    :)
    if ($conceptItemType) then (
        switch (lower-case($conceptItem))
        case 'group'
        case 'display'
        case 'question'
        case 'boolean'
        case 'decimal'
        case 'integer'
        case 'date'
        case 'time'
        case 'string'
        case 'text'
        case 'url'
        case 'choice'
        case 'open-choice'
        case 'attachment'
        case 'reference'
        case 'quantity' return $conceptItem
        case 'datetime' return 'dateTime'
        default return error(xs:QName('quest:IllegalItemType'), concat('Concept id=', $concept/@id, ' effectiveDate=', $concept/@effectiveDate, ' ', $concept/name[1], ' defines unknown property Quest::ItemType value: ''', $conceptItem, ''''))
    )
    (: Input param is a concept, since we need to look at group :)
    else if ($concept/@type = 'group') then 'group' 
    else (
        switch ($valueDomainType)
        case 'boolean'
        case 'date'
        case 'decimal'
        case 'quantity'
        case 'text'
        case 'time' return $valueDomainType
        case 'count' return 'integer'
        case 'datetime' return 'dateTime'
        case 'duration' return 'quantity'
        case 'currency' return 'quantity'
        case 'identifier' return 'string'
        case 'ordinal' return 'choice'
        case 'blob' return 'attachment'
        case 'code' return 
            if ($concept/*:valueSet) then
                if ($concept/*:terminologyAssociation[@strength[not(. = 'required')]]) then 'open-choice'
                else if ($concept/*:valueSet[*:completeCodeSystem | *:conceptList/*:include | *:conceptList/*:exclude]) then 'open-choice'
                else if ($concept/*:valueSet/*:conceptList/*[@code = 'OTH'][@codeSystem = '2.16.840.1.113883.65.1008']) then 'open-choice'
                else 'choice'
            else 'string'
        default return  'string'
    )
};

declare %private function utilsc:decor2questionnaireAnswerType($concept as element(concept)) as xs:string {
    
    (: Input param is a concept, since we need to look at group :)
    switch ($concept/valueDomain/@type)
    case 'count' return 'valueInteger'
    case 'date' return 'valueDate'
    case 'time'  return 'valueTime'
    case 'code' return 'valueCoding'
    case 'blob' return 'valueReference'
    default return 'valueString'
};

declare %private function utilsc:mergeDatasets($savedTransaction as element(transaction), $mergeDataset as element(dataset), $conceptId as xs:string?, $conceptEffectiveDate as xs:string?) as element(dataset) {
    
    if (empty($conceptId)) then (
        (: get full thing :)
        element {name($mergeDataset)} {
            $mergeDataset/(@* except (@transactionId | @transactionEffectiveDate | @templateId | @templateEffectiveDate)),
            attribute transactionId {$savedTransaction/@id},
            attribute transactionEffectiveDate {$savedTransaction/@effectiveDate},
            if ($savedTransaction[representingTemplate/@ref]) then (
                attribute templateId {$savedTransaction/representingTemplate/@ref},
                if ($savedTransaction[representingTemplate/@flexibility]) then
                    attribute templateEffectiveDate {$savedTransaction/representingTemplate/@flexibility}
                else ()
            ) else (),
            $mergeDataset/node()
        }
    )
    else (
        let $savedDataset   := utillib:getDatasetTree((), (), $savedTransaction/@id, $savedTransaction/@effectiveDate, true())
        return
        element {name($mergeDataset)} {
            $mergeDataset/(@* except (@transactionId | @transactionEffectiveDate | @templateId | @templateEffectiveDate)),
            attribute transactionId {$savedTransaction/@id},
            attribute transactionEffectiveDate {$savedTransaction/@effectiveDate},
            if ($savedTransaction[representingTemplate/@ref]) then (
                attribute templateId {$savedTransaction/representingTemplate/@ref},
                if ($savedTransaction[representingTemplate/@flexibility]) then
                    attribute templateEffectiveDate {$savedTransaction/representingTemplate/@flexibility}
                else ()
            ) else (),
            for $concept in $savedDataset/node()
            return utilsc:mergeConcepts($concept, $mergeDataset//concept[@id = $conceptId][@effectiveDate = $conceptEffectiveDate])
        }
    )
};

declare %private function utilsc:mergeConcepts($concept as item(), $mergeConcept as element(concept)) as item() {
    
    if ($concept[@id = $mergeConcept/@id][@effectiveDate = $mergeConcept/@effectiveDate]) then
        $mergeConcept
    else if ($concept/descendant::concept[@id = $mergeConcept/@id][@effectiveDate = $mergeConcept/@effectiveDate]) then (
        element concept {
            $concept/(@* except @absent),
            for $subconcept in $concept/node()
            return utilsc:mergeConcepts($subconcept, $mergeConcept)
        }
    )
    else ($concept)
};

declare %private function utilsc:checkConceptConstraint($dataset as element(dataset)) {

    let $coconstraint           := $dataset//(concept | condition)[@minimumMultiplicity[not(matches(., '^(\?|0|([1-9]\d*))?$'))] | @maximumMultiplicity[not(matches(., '^(\?|\*|0|([1-9]\d*))?$'))]]
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A concept or condition minimumMultiplicity SHALL be empty, ? or numeric, and maximumMultiplicity SHALL be empty, ?, * or numeric. Found: ', string-join(
                for $c in $coconstraint
                let $concept := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1], $c/@conformance), '')
            , ', ')))
        else ()
    
    let $coconstraint           := $dataset//(concept | condition)[@conformance[. = 'M'] | @isMandatory[. = 'true']][@minimumMultiplicity[. = '0'] | @maximumMultiplicity[. = '0'] | @conformance[not(. = ('', 'R', 'M'))]]
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A mandatory concept or condition SHALL NOT have minimum or maximum cardinality 0 or conformance other than R(equired). Found: ', string-join(
                for $c in $coconstraint
                let $concept    := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((concat(($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1]), $c/@conformance, if ($c[not($c/@conformance)][@isMandatory = 'true']) then 'M' else ()), ' ')
            , ', ')))
        else ()
        
    let $coconstraint           := $dataset//(concept | condition)[@conformance = 'NP'][@minimumMultiplicity[. != '0'] | @maximumMultiplicity[. != '0']]
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A concept or condition with conformance N(ot)P(resent) SHALL NOT have minimum or maximum cardinality other than 0. Found: ', string-join(
                for $c in $coconstraint
                let $concept := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1], $c/@conformance), '')
            , ', ')))
        else ()
        
    let $coconstraint           := $dataset//concept[@conformance = 'C'][empty(condition) and empty(enableWhen)]
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A conditional concept SHALL have one or more conditions / enableWhen. Found: ', string-join(
                for $c in $coconstraint
                let $concept := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1], $c/@conformance), '')
            , ', ')))
        else ()
    
    let $coconstraint           := $dataset//concept[not(@conformance = 'C')][condition]
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A concept with conditions SHALL have conformance C. Found: ', string-join(
                for $c in $coconstraint
                let $concept := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1], $c/@conformance), '')
            , ', ')))
        else ()
    
    let $coconstraint           := $dataset//condition[@conformance = 'C']
    let $check                  :=
        if ($coconstraint) then
            error($errors:BAD_REQUEST, concat('A concept condition SHALL NOT have conformance C. Found: ', string-join(
                for $c in $coconstraint
                let $concept := $c/ancestor-or-self::concept[1]
                return
                    '''' || $concept/name[1] || ''' id ''' || tokenize($concept/(@ref | @id), '\.')[last()] || ''' ' || string-join((($c/@minimumMultiplicity[not(. = '')], '?')[1], '..', ($c/@maximumMultiplicity[not(. = '')], '?')[1], $c/@conformance), '')
            , ', ')))
        else ()
        
    return ()   
};