xquery version "3.1";
(:
    ART-DECOR® STANDARD COPYRIGHT AND LICENSE NOTE
    Copyright © ART-DECOR Expert Group and ART-DECOR Open Tools GmbH
    see https://docs.art-decor.org/copyright and https://docs.art-decor.org/licenses

    This file is part of the ART-DECOR® tools suite.
:)
(:    
    enumeration:
      noAssociation              No terminology association defined
      noValuesetAssociated       No valueset associatiated with conceptlist
      noValuesetItem             No valueset item associated with concept in conceptlist
      noUcumUnit                       The unit of the value domain is not a valid UCUM essence unit
      conceptNotFound            Concept not found in codesystem
      codesystemNotfound         Codesystem not found on server
      valuesetNotFound           Valueset not found
      valueSetSizeNotConceptListSize   The linked value set is not of the same size as the concept list
      noMatchingDesignation      Display name does not match any designation in concept
      designationCaseMismatch    Case of display name does not match designation case in concept
      conceptRetired             Concept is retired
      conceptDraft               Concept is in draft
      conceptExperimental        Concept is experimental
      cannotParseExpression      Postcoordinated expression cannot be parsed
      equalSignNotAtStart        equal sign is not allowed at start of a postcoordinated code
      uriNotValidCode            URI is not a valid code
      errorInExpression          Error in Snomed expression
      notCoreModule                    Concept is not part of Snomed core module
      ok                         OK
      
   enumeration severity:
      info
      warning
      error
:)
(:~ Terminology Report API creates Terminology Reports for a Project (CADTS-aware) :)
module namespace trepapi            = "http://art-decor.org/ns/api/terminology/report";


import module namespace roaster     = "http://e-editiones.org/roaster";
import module namespace errors      = "http://e-editiones.org/roaster/errors";

import module namespace treplib     = "http://art-decor.org/ns/api/terminology/reportlib" at "library/terminology-report-lib.xqm";
import module namespace utillib      = "http://art-decor.org/ns/api/util" at "library/util-lib.xqm";
import module namespace setlib    = "http://art-decor.org/ns/api/settings" at "library/settings-lib.xqm";
import module namespace decorlib      = "http://art-decor.org/ns/api/decor" at "library/decor-lib.xqm";

declare namespace json      = "http://www.json.org";
declare namespace rest      = "http://exquery.org/ns/restxq";
declare namespace resterr   = "http://exquery.org/ns/restxq/error";
declare namespace http      = "http://expath.org/ns/http-client";
declare namespace output    = "http://www.w3.org/2010/xslt-xquery-serialization";


declare %private variable $trepapi:PROGRESS-REPORT-10                := 'Checking request parameters ...';
declare %private variable $trepapi:PROGRESS-DATASET-20               := 'Checking datasets ...';
declare %private variable $trepapi:PROGRESS-SCENARIO-40              := 'Checking scenarios';
declare %private variable $trepapi:PROGRESS-VALUESET-60              := 'Checking valuesets ...';




(:~ Get a terminology report (list) for a given project
    @param $projectPrefix     - required path part which may be its prefix or its oid
    @return as JSON
    @since 2022-11-30
:)
declare function trepapi:getTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    let $retrieveReport       := $request?parameters?retrieve = true()
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    let $check                  :=
        if (empty($specificref)) then () else if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $results                := trepapi:getTerminologyReport($projectPrefix,$specificref,$retrieveReport)
     
    return
        if (empty($results)) then 
            roaster:response(404, ())
        else (
            $results
        )
};

(:~ Get a terminology report (list) for a given project
    @param $project                 - required path part which may be its prefix or its oid
    @param $projectLanguage         - optional parameter to select from a specific language
    @param $format                  - optional. if not given it is html
    @param $download                - optional as xs:boolean. Default: false. 
    @return as html
    @since 2025-06-19
:)
declare function trepapi:getTerminologyReportView($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $projectLanguage        := $request?parameters?language[not(. = '')]
    let $reportId               := $request?parameters?reportId[string-length() gt 0]
    let $format                 := $request?parameters?format[not(. = '')]
    let $download               := $request?parameters?download=true()
    
    let $format                         :=
    if (not($format = 'list')) then 'html' else 'list'
    
(:    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()   :)
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $terminologyReport      := $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $reportId]
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
(:    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        ) :)
    let $check                  :=
        if (empty($reportId)) then () else if (matches($reportId, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter reportId SHALL be an oid or uuid. Found: ' || $reportId)
        )
    
    
    let $filename                       := 'TR_' || $projectPrefix || '_(download_' || substring(fn:string(current-dateTime()),1,19) || ').html' 
    let $r-header                       := 
        (response:set-header('Content-Type', 'text/html; charset=utf-8'),
        if ($download) then response:set-header('Content-Disposition', 'attachment; filename='|| $filename) else ())

    return
        if (empty($terminologyReport)) then (
            roaster:response(404, ())
        )
        else 
        
        (: prepare for Html :)
        let $referenceUrl               := $decor/project/reference[@url castable as xs:anyURI]/@url
                    
        return 
            if ($format = 'list') 
                then treplib:convertReportSimple2Html($terminologyReport, $projectLanguage, $referenceUrl, $download) 
                else treplib:convertReport2Html($terminologyReport, $projectLanguage, $referenceUrl, $download)

};


(:~ Create new terminology report and return the listing of terminology reports

@param $project - required. project prefix or id of the project to create the environment for
@return xml listing of terminology reports
:)
declare function trepapi:createTerminologyReportRequest($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (empty($request?body)) then 
            error($errors:BAD_REQUEST, 'Request SHALL have data')
        else ()
        
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if ($decor) then () else (
            error($errors:BAD_REQUEST, concat('Project with prefix or id ''', $project, ''', does not exist.'))
        )
    let $check                  :=
        if (decorlib:authorCanEditP($authmap, $decor, $decorlib:SECTION-PROJECT)) then () else (
            error($errors:FORBIDDEN, concat('User ', $authmap?name, ' does not have sufficient permissions to create a runtime version of project ', $project, '. You have to be an active author in the project.'))
        )
    
    let $now                    := substring(xs:string(current-dateTime()), 1, 19)
    let $stamp                  := util:uuid()
    let $author                 := $decor/project/author[@username = $authmap?name]
    let $author                 := if (empty($author)) then $authmap?name else $author
    let $language               := $decor/project/@defaultLanguage
    let $timeStamp              := 'development'
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
        
    
    let $terminology-report-request   :=   <terminology-report-request uuid="{$stamp}" for="{$projectPrefix}" on="{$now}" as="{$stamp}" by="{$author}" language="{$language}" progress="Added to process queue ..." progress-percentage="0"/>

    let $write                  := xmldb:store($setlib:strDecorScheduledTasks, $projectPrefix || replace($now, '\D', '') || '.xml', $terminology-report-request)
    let $tt                     := sm:chmod(xs:anyURI($write), 'rw-rw----')
    
    return
        trepapi:getTerminologyReport($projectPrefix, (),())
};

(:~ 
   Get list of terminology reports or specific report

   @param $project - required. project prefix or id of the project
   @return list of terminology reports or specific report
:)
declare function trepapi:getTerminologyReport($projectPrefix as xs:string*, $specificref as xs:string?, $retrieve as xs:boolean?) {
   if ($retrieve) then (
        let $fullReport := $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
      return
      $fullReport
      )
    else (
        let $reportrequestF        :=
            if (string-length($specificref) > 0)
            then
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
            else
                collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix]
        let $terminologyReportF              :=
            if (string-length($specificref) > 0)
            then
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix][@as = $specificref]
            else
                $setlib:colDecorVersion//terminologyReport[@for = $projectPrefix]
    
        let $count                  := count($terminologyReportF | $reportrequestF)
        
        let $terminologyReportF              :=
            for $c in $terminologyReportF
            let $as   := $c/@as
            group by $as
            order by $c/@on descending
            return $c[1]
        
        let $results                :=
            <list artifact="TERMINOLOGYREPORT" current="{$count}" total="{$count}" all="{$count}" lastModifiedDate="{current-dateTime()}">
            {
                for $c in $reportrequestF
                order by $c/@on descending
                return
                    element {name($c)}
                    {
                        $c/@*
                    }
                ,
                for $c at $count in $terminologyReportF
                return
                    <terminologyReport>
                    {
                        $c/@* except $c/@status,
                        if ($count < 3) then $c/@status else attribute {'status'} {'retired'},
                        $c/filter
                    }
                    </terminologyReport>
            }
            </list>
    
        for $result in $results
        return
            element {name($result)} {
                $result/@*,
                namespace {"json"} {"http://www.json.org"},
                utillib:addJsonArrayToElements($result/*)
            }
    )
};



(:~ Process terminology report requests :)
declare function trepapi:process-terminology-report-request($threadId as xs:string, $request as element(terminology-report-request)) {   
    let $xsltParameters     :=  <parameters></parameters>
    let $logsignature           := 'trepapi:process-terminology-report-request'
    let $mark-busy              := update insert attribute busy {'true'} into $request
    let $progress               := 
        if ($request/@progress) then (
            update value $request/@progress with $trepapi:PROGRESS-REPORT-10 
        )
        else (
            update insert attribute progress {$trepapi:PROGRESS-REPORT-10} into $request
        )
    let $progress               := 
        if ($request/@progress-percentage) then (
            update value $request/@progress-percentage with round((100 div 9) * 1) 
        )
        else (
            update insert attribute progress-percentage {round((100 div 9) * 1)} into $request
        )
    let $timeStamp              := current-dateTime()
    let $projectPrefix          := $request/@for[not(. = '*')]
    let $decor                  := utillib:getDecor($projectPrefix)
    let $language               := $request/@language[not(. = '')]
    let $reportCollection       := concat($setlib:strDecorVersion,'/',substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')

    return
        if (count($decor) = 1) then (
            
            update value $request/@progress with $trepapi:PROGRESS-DATASET-20,
            update value $request/@progress-percentage with round((100 div 9) * 2),
            (: check if CADTS codesystems are available :)
          (:  let $codesystemCheck :=
                  for $codesystem in $decor/terminology/codeSystem
                  return
                  if (collection(concat($setlib:strCodesystemStableData,'/projects'))//browsableCodeSystem[@oid=$codesystem/@id][@effectiveDate=$codesystem/@effectiveDate]) then
                     'true'
                  else 'false'
            let $convert :=
                  if ($codesystemCheck='false') then
                     'test'
                  else() :)
            let $filteredDecor  := treplib:getFilteredDecor($decor)
            let $terminologyReport := 
                <terminologyReport  for="{$request/@for}" on="{$request/@on}" as="{$request/@as}" by="{$request/@by}" status="active">
                    <project>
                     {$decor/project/name}
                     </project>
                    {
                     $filteredDecor/filter
                     }
                    <datasets datasetCount="{count($filteredDecor/datasets/dataset)}">
                    {
                        treplib:traverseDatasets($filteredDecor)
                    }
                    </datasets>
                {
                    update value $request/@progress with $trepapi:PROGRESS-SCENARIO-40,
                    update value $request/@progress-percentage with round((100 div 9) * 4)
                }
                    <transactions transactionCount="{count($filteredDecor/scenarios/scenario/transaction/transaction[representingTemplate][@statusCode=('draft','final','pending')])}">
                    {
                        (: datasets may have versions and concepts may have versions. we need to use both id and effectiveDate to get to the dataset. within a dataset there can be only 1 version of a concept :)
                        for $transaction in $filteredDecor/scenarios/scenario/transaction/transaction[representingTemplate][@statusCode=('draft','final','pending')]
                        let $ds  := if ($transaction/representingTemplate/@sourceDataset) then utillib:getDataset($transaction/representingTemplate/@sourceDataset, $transaction/representingTemplate/@sourceDatasetFlexibility) else ()
                        return
                        <transaction json:array="true">
                        {
                            $transaction/@*,
                            $transaction/name,
                            <representingTemplate conceptCount="{count($transaction/representingTemplate//concept)}">
                            {
                                $transaction/representingTemplate/@*
                            }
                                <dataset>
                                {
                                    $ds/name
                                }
                                </dataset>
                            {
                                for $concept in $transaction/representingTemplate/concept
                                return
                                    for $association in $concept/terminologyAssociation
                                        let $sourceconcept := ($ds//concept[@id=$concept/@ref])[1]
                                        return
                                            if ($association/@conceptId=$concept/@ref) then
                                    (: check concept association :)
                                      <concept json:array="true">
                                            {
                                                $sourceconcept/@*,
                                                $sourceconcept/name,
                                                treplib:handleTerminologyAssociation($filteredDecor, $concept/terminologyAssociation)
                                            }
                                            </concept>
                                    else if ($association/@conceptId=$sourceconcept/conceptList/@id) then
                                       (: check conceptList association :)
                                       treplib:handleConceptListAssociation($filteredDecor,$association)
                                    else if ($association/@conceptId=$sourceconcept/conceptList/concept/@id) then
                                       (: check conceptList concept associaation :)
                                       treplib:handleTerminologyAssociation($filteredDecor,$association)
                                    else ()
                            
                            }
                            </representingTemplate>
                        }
                        </transaction>
                    }
                    </transactions>
                {
                    update value $request/@progress with $trepapi:PROGRESS-VALUESET-60,
                    update value $request/@progress-percentage with round((100 div 9) * 6)
                }
                    <valueSets valueSetCount="{count($filteredDecor/terminology/valueSet[@id])}">
                    {
                        treplib:traverseValuesets($filteredDecor)   
                    }
                    </valueSets>
                    <ids>
                    {
                        let $codeSystemIds := distinct-values($decor/terminology/terminologyAssociation/@codeSystem | $decor/terminology/valueSet/conceptList/concept/@codeSystem)
                        let $idCheck :=
                            for $codeSystemId in $codeSystemIds
                            let $id := $decor/ids/id[@root = $codeSystemId]
                            return
                                <id>
                                {
                                    if ($id) then ($id/@*, $id/*) else (
                                        <message severity="warning" type="idRefMissing" codeSystem="{$codeSystemId}"></message>
                                    )
                                }
                                </id>
                        return
                            $idCheck
                    }
                    </ids>
                </terminologyReport>
            return (
                if (not(xmldb:collection-available($reportCollection))) then
                    try {
                        let $ttt    := xmldb:create-collection($setlib:strDecorVersion, concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development'))
                        let $tt     := try { sm:chmod(xs:anyURI($ttt), 'rwxrwsr-x') } catch * {()}
                        return $ttt
                    } 
                    catch * {
                        <error>+++ Create collection failed '{concat(substring($projectPrefix, 1, string-length($projectPrefix) - 1),'/development')}': {$err:code}: {$err:description}</error>
                    }
                else()
                ,
                xmldb:store($reportCollection,concat($projectPrefix,'terminology-report-',$request/@as,'.xml'),transform:transform($terminologyReport ,xs:anyURI(concat('xmldb:exist://',$setlib:strApi ,'/resources/stylesheets/add-metadata-to-terminologyReport.xsl')), $xsltParameters))
                ,
                xmldb:remove(util:collection-name($request), util:document-name($request))
            )                
        )
        else (
            let $message                := 'ERROR. Found ' || count($decor) || ' projects for prefix ' || $projectPrefix || '. Expected: 1. Compile id: ' || $request/@as
            let $log                    := utillib:log-scheduler-event('error', $threadId, $logsignature, $message)
            let $progress               := update value $request/@progress with $message
            
            return ()
        )
    

};


(:~ Deletes terminology report

@param $project - required. project prefix or id of the project
@param $specificref - required. reference to the runtime environmnent
@return xml listing of environments
:)
declare function trepapi:deleteTerminologyReport($request as map(*)) {
    
    let $authmap                := $request?user
    let $project                := $request?parameters?project[string-length() gt 0]
    let $specificref            := $request?parameters?reportId[string-length() gt 0]
    
    let $check                  :=
        if (empty($authmap)) then 
            error($errors:UNAUTHORIZED, 'You need to authenticate first')
        else ()
    let $check                  :=
        if (empty($project)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter project')
        else ()
    let $check                  :=
        if (matches($specificref, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter compilationId SHALL be an oid or uuid. Found: ' || $specificref)
        )
    
    let $decor                  := utillib:getDecor($project)
    let $projectPrefix          := $decor/project/@prefix
    
    let $check                  :=
        if (count($projectPrefix) = 1) then () else (
            error($errors:BAD_REQUEST, 'Project parameter ' || $project || ' SHALL refer to exactly one project. Found: ' || count($decor))
        )
    
    let $part1                  := concat('xmldb:exist://', $setlib:strDecorVersion, '/', substring($projectPrefix, 1, string-length($projectPrefix) - 1), '/development/')
    let $tmp1                   := 
        if (xmldb:collection-available($part1)) then (
            if (doc-available(concat($part1, $projectPrefix,'terminology-report-', $specificref, '.xml'))) then xmldb:remove($part1, concat($projectPrefix,'terminology-report-', $specificref, '.xml')) else ()
        )
        else () 
    
    let $part1                  := collection($setlib:strDecorScheduledTasks)/terminology-report-request[@for = $projectPrefix][@as = $specificref]
    let $tmp3                   := if ($part1) then xmldb:remove(util:collection-name($part1), util:document-name($part1)) else ()
    
    return
        roaster:response(204, ())
};


(:~ Validate a given code against a codesystem
    @param $projectPrefix     - required path part which may be its prefix or its oid
    @return as JSON
    @since 2023-06-27
:)
declare function trepapi:validateCode($request as map(*)) {
    
    let $data                 := utillib:getBodyAsXml($request?body, (), ())
    let $codeSystemId         :=  $data/@codeSystemId
    let $code                 := $data/@code
    let $displayName          := $data/@displayName
   
    let $check                  :=
        if (empty($codeSystemId)) then () else if (matches($codeSystemId, '^([1-9][0-9]*(\.[0-9]+)*)|([0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12})$', 'i')) then () else (
            error($errors:BAD_REQUEST, 'Parameter codeSystemId SHALL be an oid or uuid. Found: ' || $codeSystemId)
        )
        
    let $check                  :=
        if (empty($code)) then
            error($errors:BAD_REQUEST, 'You are missing required parameter code')
        else ()

    let $concept :=
         <concept code="{$code}" codeSystem="{$codeSystemId}" displayName="{$displayName}">
            {
            if (starts-with($code,'http')) then
               <message severity="error" type="uriNotValidCode" json:array="true">URI is not a valid code</message>
            else
            (: handle SNOMED post coordinated codes :)
                if ($codeSystemId='2.16.840.1.113883.6.96') then
                    if ($code castable as xs:integer) then
                        let $snomedConcept := collection($setlib:strCodesystemStableData)//concept[@code=$code][ancestor::browsableCodeSystem/@oid=$codeSystemId]
                        return
                        if ($snomedConcept) then
                            treplib:checkValueSetConcept($displayName,$snomedConcept)
                        else
                            (: check if code system is present :)
                            if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$codeSystemId) then
                                <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$codeSystemId]/name}</codeSystem></message>
                            else
                            <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
                    else 
                        treplib:checkSnomedExpression($code,$displayName)
                else
                     (: find code system first, local or cadts :)
                     if (collection($setlib:strCodesystemStableData)//browsableCodeSystem/@oid=$codeSystemId) then
                        let $codeSystemConcept := collection($setlib:strCodesystemStableData)//concept[@code=$code][ancestor::browsableCodeSystem/@oid=$codeSystemId]
                        return
                        if ($codeSystemConcept) then
                           treplib:checkValueSetConcept($displayName,$codeSystemConcept)
                        else
                           <message severity="warning" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strCodesystemStableData)//browsableCodeSystem[@oid=$codeSystemId]/name}</codeSystem></message>
                     else if (collection($setlib:strDecorData)//codeSystem[@id=$codeSystemId]) then
                        let $localCodeSystemConcept := collection($setlib:strDecorData)//codedConcept[@code=$code]
                        return
                        if ($localCodeSystemConcept) then
                           let $localConcept :=
                              <concept>
                                 {
                                 $localCodeSystemConcept/@*,
                                 for $designation in $localCodeSystemConcept/designation
                                 return
                                 <designation lang="{$designation/@language}" type="{if ($designation/@type='preferred') then 'pref' else 'syn'}">{$designation/@displayName/string()}</designation>
                                 }
                              </concept>
                           return
                           treplib:checkValueSetConcept($displayName,$localConcept)
                        else
                           <message severity="error" type="conceptNotFound" json:array="true"><codeSystem oid="{$codeSystemId}">{collection($setlib:strDecorData)//codeSystem[@id=$codeSystemId]/@displayName}</codeSystem></message>
                     else
                        <message severity="warning" type="codesystemNotfound" json:array="true">Codesystem not found</message>
            }
         </concept>
     
    return
        if (empty($concept)) then 
            roaster:response(404, ())
        else (
            $concept
        )
};
